package org.jdesktop.animation.timing;

import java.awt.event.ActionEvent;
import java.awt.Toolkit;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import javax.swing.Timer;

public class TimingController implements TimingTarget
{

    public enum Direction
    {
        FORWARD, 
        BACKWARD;
    }
    
    private class TimerTarget implements ActionListener
    {
        public void actionPerformed(final ActionEvent e) {
            final long currentTime = System.nanoTime() / 1000000L;
            final long cycleElapsedTime = currentTime - TimingController.this.currentStartTime;
            final long totalElapsedTime = currentTime - TimingController.this.startTime;
            final double currentCycle = totalElapsedTime / (double)TimingController.this.cycle.getDuration();
            if (TimingController.this.envelope.getRepeatCount() != -1.0 && currentCycle >= TimingController.this.envelope.getRepeatCount()) {
                switch (TimingController.this.envelope.getEndBehavior()) {
                    case HOLD: {
                        float endFraction;
                        if (TimingController.this.intRepeatCount) {
                            if (TimingController.this.direction == Direction.BACKWARD) {
                                endFraction = 0.0f;
                            }
                            else {
                                endFraction = 1.0f;
                            }
                        }
                        else {
                            endFraction = Math.min(1.0f, cycleElapsedTime / (float)TimingController.this.cycle.getDuration());
                        }
                        TimingController.this.timingEventPreprocessor(cycleElapsedTime, totalElapsedTime, endFraction);
                        break;
                    }
                    case RESET: {
                        TimingController.this.timingEventPreprocessor(cycleElapsedTime, totalElapsedTime, 0.0f);
                        break;
                    }
                }
                TimingController.this.stop();
            }
            else if (TimingController.this.cycle.getDuration() != -1 && cycleElapsedTime > TimingController.this.cycle.getDuration()) {
                final long actualCycleTime = cycleElapsedTime % TimingController.this.cycle.getDuration();
                float fraction = actualCycleTime / (float)TimingController.this.cycle.getDuration();
                TimingController.this.currentStartTime = currentTime - actualCycleTime;
                if (TimingController.this.envelope.getRepeatBehavior() == Envelope.RepeatBehavior.REVERSE) {
                    final boolean oddCycles = (int)(cycleElapsedTime / TimingController.this.cycle.getDuration()) % 2 > 0;
                    if (oddCycles) {
                        TimingController.this.direction = ((TimingController.this.direction == Direction.FORWARD) ? Direction.BACKWARD : Direction.FORWARD);
                    }
                    if (TimingController.this.direction == Direction.BACKWARD) {
                        fraction = 1.0f - fraction;
                    }
                }
                TimingController.this.timingEventPreprocessor(actualCycleTime, totalElapsedTime, fraction);
                TimingController.this.repeat();
            }
            else {
                float fraction2 = 0.0f;
                if (TimingController.this.cycle.getDuration() != -1) {
                    fraction2 = cycleElapsedTime / (float)TimingController.this.cycle.getDuration();
                    if (TimingController.this.direction == Direction.BACKWARD) {
                        fraction2 = 1.0f - fraction2;
                    }
                    fraction2 = Math.min(fraction2, 1.0f);
                    fraction2 = Math.max(fraction2, 0.0f);
                }
                TimingController.this.timingEventPreprocessor(cycleElapsedTime, totalElapsedTime, fraction2);
            }
        }
    }

    private Timer timer;
    private ArrayList<TimingTarget> targets;
    private long startTime;
    private long currentStartTime;
    private int currentCycle;
    private Direction direction;
    private boolean intRepeatCount;
    private ArrayList<TimingListener> listeners;
    private Envelope envelope;
    private Cycle cycle;
    private float acceleration;
    private float deceleration;
    public static final int INFINITE = -1;
    
    public TimingController(final Cycle cycle, final Envelope envelope) {
        this(cycle, envelope, null);
    }
    
    public TimingController(final Cycle cycle, final Envelope envelope, final TimingTarget target) {
        this.targets = new ArrayList <TimingTarget>();
        this.currentCycle = 0;
        this.direction = Direction.FORWARD;
        this.listeners = new ArrayList<TimingListener>();
        this.acceleration = 0.0f;
        this.deceleration = 0.0f;
        this.cycle = cycle;
        this.envelope = envelope;
        if (target != null) {
            this.targets.add(target);
        }
        this.intRepeatCount = (Math.rint(envelope.getRepeatCount()) == envelope.getRepeatCount());
        final TimerTarget timerTarget = new TimerTarget();
        (this.timer = new Timer(cycle.getResolution(), timerTarget)).setInitialDelay(envelope.getBegin());
        final Toolkit tk = Toolkit.getDefaultToolkit();
    }
    
    public TimingController(final int duration, final TimingTarget target) {
        this(new Cycle(duration, 10), new Envelope(1.0, 0, Envelope.RepeatBehavior.FORWARD, Envelope.EndBehavior.HOLD), target);
    }
    
    public void setAcceleration(final float acceleration) {
        if (acceleration < 0.0f || acceleration > 1.0f) {
            throw new IllegalArgumentException("Acceleration value cannot lie outside [0,1] range");
        }
        if (acceleration > 1.0f - this.deceleration) {
            throw new IllegalArgumentException("Acceleration value cannot be greater than (1 - deceleration)");
        }
        this.acceleration = acceleration;
    }
    
    public void setDeceleration(final float deceleration) {
        if (deceleration < 0.0f || deceleration > 1.0f) {
            throw new IllegalArgumentException("Deceleration value cannot lie outside [0,1] range");
        }
        if (deceleration > 1.0f - this.acceleration) {
            throw new IllegalArgumentException("Deceleration value cannot be greater than (1 - acceleration)");
        }
        this.deceleration = deceleration;
    }
    
    public float getAcceleration() {
        return this.acceleration;
    }
    
    public float getDeceleration() {
        return this.deceleration;
    }
    
    public void addTarget(final TimingTarget target) {
        if (target != null) {
            synchronized (this.targets) {
                this.targets.add(target);
            }
        }
    }
    
    public void addTimingListener(final TimingListener listener) {
        if (listener != null) {
            this.listeners.add(listener);
        }
    }
    
    public Cycle getCycle() {
        return this.cycle;
    }
    
    public Envelope getEnvelope() {
        return this.envelope;
    }
    
    public void setCycle(final Cycle cycle) {
        this.cycle = cycle;
        this.timer.setDelay(cycle.getResolution());
    }
    
    public void setEnvelope(final Envelope envelope) {
        this.envelope = envelope;
    }
    
    public void start() {
        this.begin();
        this.startTime = System.nanoTime() / 1000000L + this.envelope.getBegin();
        this.currentStartTime = this.startTime;
        this.timer.start();
    }
    
    public boolean isRunning() {
        return this.timer.isRunning();
    }
    
    public void stop() {
        this.timer.stop();
        this.end();
    }
    
    public void timingEvent(final long cycleElapsedTime, final long totalElapsedTime, final float fraction) {
        synchronized (this.targets) {
            for (int i = 0; i < this.targets.size(); ++i) {
                final TimingTarget target = this.targets.get(i);
                target.timingEvent(cycleElapsedTime, totalElapsedTime, fraction);
            }
        }
    }
    
    public void begin() {
        synchronized (this.targets) {
            for (int i = 0; i < this.targets.size(); ++i) {
                final TimingTarget target = this.targets.get(i);
                target.begin();
            }
        }
        synchronized (this.listeners) {
            for (int i = 0; i < this.listeners.size(); ++i) {
                final TimingListener listener = this.listeners.get(i);
                listener.timerStarted(new TimingEvent(this));
            }
        }
    }
    
    public void end() {
        synchronized (this.targets) {
            for (int i = 0; i < this.targets.size(); ++i) {
                final TimingTarget target = this.targets.get(i);
                target.end();
            }
        }
        synchronized (this.listeners) {
            for (int i = 0; i < this.listeners.size(); ++i) {
                final TimingListener listener = this.listeners.get(i);
                listener.timerStopped(new TimingEvent(this));
            }
        }
    }
    
    protected void repeat() {
        synchronized (this.listeners) {
            for (int i = 0; i < this.listeners.size(); ++i) {
                final TimingListener listener = this.listeners.get(i);
                listener.timerRepeated(new TimingEvent(this));
            }
        }
    }
    
    private void timingEventPreprocessor(final long cycleElapsedTime, final long totalElapsedTime, float fraction) {
        if (this.acceleration != 0.0f || this.deceleration != 0.0f) {
            final float oldFraction = fraction;
            final float runRate = 1.0f / (1.0f - this.acceleration / 2.0f - this.deceleration / 2.0f);
            if (fraction < this.acceleration) {
                final float averageRunRate = runRate * (fraction / this.acceleration) / 2.0f;
                fraction *= averageRunRate;
            }
            else if (fraction > 1.0f - this.deceleration) {
                final float tdec = fraction - (1.0f - this.deceleration);
                final float pdec = tdec / this.deceleration;
                fraction = runRate * (1.0f - this.acceleration / 2.0f - this.deceleration + tdec * (2.0f - pdec) / 2.0f);
            }
            else {
                fraction = runRate * (fraction - this.acceleration / 2.0f);
            }
            if (fraction < 0.0f) {
                fraction = 0.0f;
            }
            else if (fraction > 1.0f) {
                fraction = 1.0f;
            }
        }
        this.timingEvent(cycleElapsedTime, totalElapsedTime, fraction);
    }
    
}
