
create or replace PACKAGE "MIGRATION_REPOSITORY" AS

PROCEDURE createAllRepositoryTables;
PROCEDURE dropAllRepositoryTables;
PROCEDURE truncateAllRepositoryTables;

PROCEDURE createRepositoryTables;
PROCEDURE dropRepositoryTables;
PROCEDURE truncateRepositoryTables;

PROCEDURE createCaptureStageTables;
PROCEDURE dropCaptureStageTables;
PROCEDURE truncateCaptureStageTables;

PROCEDURE createConvertStageTables;
PROCEDURE dropConvertStageTables;
PROCEDURE truncateConvertStageTables;

END;
/

create or replace PACKAGE BODY "MIGRATION_REPOSITORY" AS

PROCEDURE create_MD_APPLICATIONFILES
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	CREATE TABLE MD_APPLICATIONFILES 
	    ( 
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     APPLICATION_ID NUMBER  NOT NULL , 
	     APPLICATIONFILE_ID NUMBER  NOT NULL , 
	     NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     URI VARCHAR2 (4000 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (100 BYTE)  NOT NULL , 
	     STATE VARCHAR2 (100 BYTE)  NOT NULL , 
	     LANGUAGE VARCHAR2 (100 BYTE) , 
	     LOC NUMBER , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (4000 BYTE) , 
	     UPDATED_ON DATE , 
	     UPDATED_BY VARCHAR2 (4000 BYTE) 
	    ) 
	 ]');
	 
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON TABLE MD_APPLICATIONFILES IS 'Holds a tuple for each file that is being processed whether it is changed or not.'
	 ]');
	
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_APPLICATIONFILES.NAME IS 'file name  //OBJECTNAME' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_APPLICATIONFILES.URI IS 'The uri is the part of the file url after the base dir has been removed.  See MD_APPLICATION for base dir' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_APPLICATIONFILES.TYPE IS 'This will denote the type of file we have, including DIR, FILE (text), BINARY, IGNORE (where we choose to ignore files)' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_APPLICATIONFILES.STATE IS 'State will be how this file is operated on.  For example. it will be OPEN, NEW, FIXED, IGNORED, REVIEWED, COMPLETE' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_APPLICATIONFILES.LANGUAGE IS 'Language of the file that has been processed.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_APPLICATIONFILES.SECURITY_GROUP_ID IS 'APEX' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_APPLICATIONFILES.CREATED_ON IS 'APEX' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_APPLICATIONFILES.CREATED_BY IS 'APEX' 
	 ]');	
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_APPLICATIONFILES.UPDATED_ON IS 'APEX'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_APPLICATIONFILES.UPDATED_BY IS 'APEX' 
	 ]');

    DBMS_OUTPUT.PUT_LINE('Table MD_APPLICATIONFILES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_APPLICATIONFILES creation'); 
    
END create_MD_APPLICATIONFILES;

PROCEDURE create_MD_APPLICATIONS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	 CREATE TABLE MD_APPLICATIONS 
	    ( 
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     APPLICATION_ID NUMBER  NOT NULL , 
	     APPLICATION_NAME VARCHAR2 (256 BYTE) , 
	     DESCRIPTION VARCHAR2 (4000 BYTE) , 
	     BASE_DIR VARCHAR2 (4000 BYTE) , 
	     OUTPUT_DIR VARCHAR2 (4000 BYTE) , 
	     BACKUP_DIR VARCHAR2 (4000 BYTE) , 
	     INPLACE NUMBER , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_APPLICATIONS IS 'This is the base table for application projects.  It holds the base information for applications associated with a database'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_APPLICATIONS.PROJECT_ID IS 'project of the database(s) this application relates to' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_APPLICATIONS.APPLICATION_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_APPLICATIONS.APPLICATION_NAME IS 'Name of the application suite  //OBJECTNAME' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_APPLICATIONS.DESCRIPTION IS 'Overview of what the application does.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_APPLICATIONS.BASE_DIR IS 'This is the base src directory for the application.  It could be an svn checkout, a clearcase view or something similar' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_APPLICATIONS.OUTPUT_DIR IS 'This is the output directory where the scanner will present the converted files, if there are converted or modified.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_APPLICATIONS.BACKUP_DIR IS 'This is the directory in which the application files are backed up if a backup is chosen'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_APPLICATIONS.INPLACE IS 'Designates whether the changes have been made inplace, in the source directory or not' 
	 ]');

    DBMS_OUTPUT.PUT_LINE('Table MD_APPLICATIONS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_APPLICATIONS creation'); 

END create_MD_APPLICATIONS;


PROCEDURE create_MD_CATALOGS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_CATALOGS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     ORACLE_NAME VARCHAR2 (256 BYTE), 
	     DUMMY_FLAG CHAR (1 BYTE) DEFAULT 'N' , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_CATALOGS IS 'Store catalogs in this table.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CATALOGS.CONNECTION_ID IS 'Foreign key into the connections table - Shows what connection this catalog belongs to //PARENTFIELD' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CATALOGS.CATALOG_NAME IS 'Name of the catalog //OBJECTNAME'
	]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CATALOGS.DUMMY_FLAG IS 'Flag to show if this catalog is a "dummy" catalog which is used as a placeholder for those platforms that do not support catalogs.  ''N'' signifies that this is NOT a dummy catalog, while ''Y'' signifies that it is.' 
	]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CATALOGS.NATIVE_SQL IS 'THe SQL used to create this catalog' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CATALOGS.NATIVE_KEY IS 'A unique identifier used to identify the catalog at source.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CATALOGS.ORACLE_NAME IS 'Transformed name of catalog.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CATALOGS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CATALOGS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');

    DBMS_OUTPUT.PUT_LINE('Table MD_CATALOGS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_CATALOGS creation'); 

END create_MD_CATALOGS;


PROCEDURE create_MD_COLUMNS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_COLUMNS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE) NOT NULL,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     TABLE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     COLUMN_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     COLUMN_ORDER NUMBER  NOT NULL , 
	     COLUMN_TYPE VARCHAR2 (256 BYTE) , 
	     PRECISION NUMBER , 
	     SCALE NUMBER , 
	     NULLABLE CHAR (1 BYTE) , 
	     DEFAULT_VALUE VARCHAR2 (4000 BYTE) , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     DATATYPE_TRANSFORMED_FLAG CHAR (1 BYTE) , 
	     ISAUTOINC char(1 BYTE) DEFAULT 'N' NOT NULL, --Y Yes N No 
	     ISNEWID  char(1 BYTE) DEFAULT 'N' NOT NULL, --Y Yes N No 
	     AUTOINC_SEEDVALUE NUMBER(38,0),
	     AUTOINC_INCREMENTVALUE NUMBER(38,0),
	     AUTOINC_LASTVALUE NUMBER(38,0),
	     COMMENTS VARCHAR2 (4000 BYTE) ,
	     COLUMN_NAMESPACE VARCHAR2 (256 BYTE),
	     PROP001           VARCHAR2 (4000 BYTE) ,
	     PROP002           VARCHAR2 (4000 BYTE) ,
	     PROP003           VARCHAR2 (4000 BYTE) ,
	     PROP004           VARCHAR2 (4000 BYTE) ,
	     PROP005           VARCHAR2 (4000 BYTE) ,
	     PROP006           VARCHAR2 (4000 BYTE) ,
	     PROP007           VARCHAR2 (4000 BYTE) ,
	     PROP008           VARCHAR2 (4000 BYTE) ,
	     PROP009           VARCHAR2 (4000 BYTE) ,
	     PROP010           VARCHAR2 (4000 BYTE) ,
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE 
	    )
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_COLUMNS IS 'Column information is stored in this table.'
	 ]');
	EXECUTE IMMEDIATE (q'[
	  COMMENT ON COLUMN MD_COLUMNS.PROJECT_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.CONNECTION_ID IS 'The table that this column is part of //PARENTFIELD' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.COLUMN_NAME IS 'The name of the column //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.COLUMN_ORDER IS 'The order this appears in the table' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.COLUMN_TYPE IS 'The type of the column'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.PRECISION IS 'The precision on the column'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.SCALE IS 'The scale of the column' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.NULLABLE IS 'Yes or No.  Null signifies NO'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.DEFAULT_VALUE IS 'Default value on the column' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.NATIVE_SQL IS 'The SQL used to create this column at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.NATIVE_KEY IS 'Unique identifier for this object at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.DATATYPE_TRANSFORMED_FLAG IS 'This is set to ''Y'' to show if the data type was transformed.  This is useful so we don''t apply more than 1 datatype transformation to a column' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.ISAUTOINC IS 'Whether auto increment/identity or not --Y Yes N No'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.ISNEWID  IS 'whether (newid()) or not i.e. guid WARNING this should be a PROP<n> field - SQLServer specific'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.AUTOINC_SEEDVALUE IS 'Autoincrement seed value null when no autoincrement'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.AUTOINC_INCREMENTVALUE IS 'Autoincrement increment value null when no autoincrement'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.AUTOINC_INCREMENTVALUE IS 'Autoincrement increment value null when no autoincrement'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.AUTOINC_LASTVALUE IS 'Autoincrement last value null when no autoincrement' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.ORACLE_NAME IS 'Transformed name of column.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_COLUMNS.COLUMN_NAMESPACE IS 'Shared Column Namespace(within a Table) for CONVERTED type'
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_COLUMNS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_COLUMNS creation'); 

END create_MD_COLUMNS;


PROCEDURE create_MD_CONNECTIONS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_CONNECTIONS 
	    ( 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     PROJECT_ID_FK NUMBER  NOT NULL , 
	     TYPE VARCHAR2 (256 BYTE) , 
	     HOST VARCHAR2 (4000 BYTE) , 
	     PORT NUMBER , 
	     USERNAME VARCHAR2 (256 BYTE) , 
	     PASSWORD VARCHAR2 (4000 BYTE) , 
	     DBURL VARCHAR2 (4000 BYTE) , 
	     NAME VARCHAR2 (255 BYTE) , 
	     NATIVE_SQL CLOB , 
	     STATUS VARCHAR2 (30 BYTE) , 
	     NUM_CATALOGS NUMBER , 
	     NUM_COLUMNS NUMBER , 
	     NUM_CONSTRAINTS NUMBER , 
	     NUM_GROUPS NUMBER , 
	     NUM_ROLES NUMBER , 
	     NUM_INDEXES NUMBER , 
	     NUM_OTHER_OBJECTS NUMBER , 
	     NUM_PACKAGES NUMBER , 
	     NUM_PRIVILEGES NUMBER , 
	     NUM_SCHEMAS NUMBER , 
	     NUM_SEQUENCES NUMBER , 
	     NUM_STORED_PROGRAMS NUMBER , 
	     NUM_SYNONYMS NUMBER , 
	     NUM_TABLES NUMBER , 
	     NUM_TABLESPACES NUMBER , 
	     NUM_TRIGGERS NUMBER , 
	     NUM_USER_DEFINED_DATA_TYPES NUMBER , 
	     NUM_USERS NUMBER , 
	     NUM_VIEWS NUMBER , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_CONNECTIONS IS 'This table is used to store connection information.  For example, in migrations, we could be carrying out a consolidation which occurs across many connections.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTIONS.CONNECTION_ID IS 'Primary key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTIONS.PROJECT_ID_FK IS 'The project to which this connection belongs //PARENTFIELD' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTIONS.TYPE IS 'The type of the connection - For example it could be used to store "ORACLE" or "MYSQL"'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTIONS.HOST IS 'The host to which this connection connects.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTIONS.PORT IS 'The port to which this connection connects'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTIONS.USERNAME IS 'The username used to make the connection' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTIONS.PASSWORD IS 'The password used to make this connection' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTIONS.DBURL IS 'The database url used to make this connection' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTIONS.NAME IS '//OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTIONS.NATIVE_SQL IS 'The native sql used to create this connection'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	 COMMENT ON COLUMN MD_CONNECTIONS.STATUS IS 'Status of Migration, = captured,converted,generated,datamoved' 
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_CONNECTIONS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_CONNECTIONS creation'); 

END create_MD_CONNECTIONS;


PROCEDURE create_MD_CONNECTION_LINKS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_CONNECTION_LINKS 
	    ( 
	     PROJECT_ID NUMBER  NOT NULL , 
	     SRC_CONNECTION_ID NUMBER  NOT NULL , 
	     TGT_CONNECTION_ID NUMBER  NOT NULL 
	    ) 
	 ]'); 

	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_CONNECTION_LINKS IS 'This links table will be the link betwen the captured and converted models.  These are mandatory links and therefore should only be inserted when both the captured and converted model is known.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTION_LINKS.PROJECT_ID IS 'Project id holding the models' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTION_LINKS.SRC_CONNECTION_ID IS 'Captured model id'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONNECTION_LINKS.TGT_CONNECTION_ID IS 'Target model connection id'
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_CONNECTION_LINKS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_CONNECTION_LINKS creation'); 

END create_MD_CONNECTION_LINKS;


PROCEDURE create_MD_CONSTRAINTS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_CONSTRAINTS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE) NOT NULL,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     TABLE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     CONSTRAINT_NAME VARCHAR2 (256 BYTE)  NOT NULL ,
	     COLUMN_NAME VARCHAR2 (256 BYTE) ,
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     DELETE_CLAUSE VARCHAR2 (4000 BYTE) , 
	     CONSTRAINT_TYPE VARCHAR2 (256 BYTE) ,
	     DETAIL_ORDER NUMBER  NOT NULL ,
	     REFCATALOG_NAME VARCHAR2 (256 BYTE) , 
	     REFSCHEMA_NAME VARCHAR2 (256 BYTE) , 
	     REFTABLE_NAME VARCHAR2 (256 BYTE) ,
	     REFCOLUMN_NAME VARCHAR2 (256 BYTE) ,
	     CONSTRAINT_TEXT CLOB , 
	     ISRULE char(1 BYTE) DEFAULT 'N' NOT NULL, --Y Yes N No
	     ISCLUSTEREDINDEX CHAR(1 BYTE) DEFAULT 'N' NOT NULL, --Y Yes N No
	     LANGUAGE VARCHAR2 (40 BYTE)  NOT NULL , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     CONSTRAINT_NAMESPACE VARCHAR2 (256 BYTE),
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 

	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_CONSTRAINTS IS 'Table for storing information about a constraint'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.PROJECT_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.CONNECTION_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.CATALOG_NAME IS 'Name of the constraint //OBJECTNAME' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.SCHEMA_NAME IS 'Name of the constraint //OBJECTNAME' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.TABLE_NAME IS 'Table on which this constraint exists //PARENTFIELD' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.CONSTRAINT_NAME IS 'Name of the constraint //OBJECTNAME' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.DELETE_CLAUSE IS 'delete option , can be either CASCADE, RESTRICT or NULL' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.CONSTRAINT_TYPE IS 'Type of the constraint (e.g. CHECK)'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.REFTABLE_NAME IS 'Used in foreign keys - this gives the table that we refer to.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.CONSTRAINT_TEXT IS 'The text of the constraint' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.LANGUAGE IS '//PUBLIC'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.ISRULE IS 'SQLSERVER has a constraint like rule that was previously flagged by an additional property. WARNING SQLSERVER specific - should be a PROP<N> field' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.ISCLUSTEREDINDEX IS 'Sybase can have clustered index.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.ORACLE_NAME IS 'Transformed name of schema.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.CONSTRAINT_NAMESPACE IS 'Shared Constraint Namespace(within a Schema) for CONVERTED type'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_CONSTRAINTS.DETAIL_ORDER IS 'Order of columns in constraint. Set to -1 if table constraint.'
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_CONSTRAINTS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_CONSTRAINTS creation'); 
 
END create_MD_CONSTRAINTS;


PROCEDURE create_MD_FILE_ARTIFACTS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_FILE_ARTIFACTS 
	    ( 
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     APPLICATION_ID NUMBER  NOT NULL , 
	     APPLICATIONFILE_ID NUMBER 
	     CONSTRAINT MD_APPL_FILE_FK_NONULL NOT NULL , 
	     ARTIFACT_ID NUMBER 
	     CONSTRAINT MD_APP_FILE_ART_NONULL NOT NULL , 
	     PATTERN VARCHAR2 (4000 BYTE) , 
	     STRING_FOUND VARCHAR2 (4000 BYTE) , 
	     STRING_REPLACED VARCHAR2 (4000 BYTE) , 
	     TYPE VARCHAR2 (200 BYTE) , 
	     STATUS VARCHAR2 (4000 BYTE) , 
	     LINE NUMBER , 
	     PATTERN_START NUMBER , 
	     PATTERN_END NUMBER , 
	     DUE_DATE DATE , 
	     DB_TYPE VARCHAR2 (100 BYTE) , 
	     CODE_TYPE VARCHAR2 (1000 BYTE) , 
	     DESCRIPTION VARCHAR2 (4000 BYTE) , 
	     PRIORITY NUMBER , 
	     SECURITY_GROUP_ID VARCHAR2 (20 BYTE) DEFAULT '0'  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (4000 BYTE) , 
	     LAST_UPDATED DATE , 
	     LAST_UPDATED_BY VARCHAR2 (4000 BYTE) 
	    ) 
	 ]'); 

	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_FILE_ARTIFACTS IS 'Holds a tuple for each interesting thing the scanner finds in a file'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_FILE_ARTIFACTS.PATTERN IS 'Pattern used to search source file for interesting artifiacts' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_FILE_ARTIFACTS.STRING_FOUND IS 'String found in source from the pattern supplied'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_FILE_ARTIFACTS.STRING_REPLACED IS 'This is the string which replace the string found if it was replaced.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_FILE_ARTIFACTS.TYPE IS 'This is the type of the replacement.  It could be a straight replace from a replacement pattern, or we could have passed the string to a translator which would change the string depending on the database.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_FILE_ARTIFACTS.STATUS IS 'Pattern used to search source file for interesting artifiacts' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_FILE_ARTIFACTS.DUE_DATE IS 'Due date is used by the TODO mechanism to manage the validation and work to complete this change' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_FILE_ARTIFACTS.DB_TYPE IS 'Source database calls type' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_FILE_ARTIFACTS.CODE_TYPE IS 'Source code db api, like dblib, jdbc' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_FILE_ARTIFACTS.DESCRIPTION IS 'This is a description of the artifact which will have a default generated by the scanner and then can be modified by the user to be more appropriate for their use' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_FILE_ARTIFACTS.PRIORITY IS 'The priority is set for the TODOs so they can be managed by the user' 
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_FILE_ARTIFACTS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_FILE_ARTIFACTS creation'); 

END create_MD_FILE_ARTIFACTS;


PROCEDURE create_MD_GROUPS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_GROUPS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE) NOT NULL,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     GROUP_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     GROUP_FLAG CHAR (1 BYTE) , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_GROUPS IS 'Groups of users in a schema'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUPS.PROJECT_ID IS 'Schema in which this object belongs //PARENTFIELD' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUPS.CONNECTION_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUPS.GROUP_NAME IS 'Name of the group //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUPS.GROUP_FLAG IS 'This is a flag to signify a group or a role.  If this is ''R'' it means the group is known as a Role.  Any other value means it is known as a group.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUPS.NATIVE_SQL IS 'SQL Used to generate this object at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUPS.NATIVE_KEY IS 'Unique id for this object at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUPS.ORACLE_NAME IS 'Transformed name of group.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUPS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUPS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_GROUPS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_GROUPS creation'); 

END create_MD_GROUPS;



PROCEDURE create_MD_GROUP_MEMBERS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_GROUP_MEMBERS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE) NOT NULL,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     GROUP_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     USERNAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_GROUP_MEMBERS IS 'This table is used to store the members of a group.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUP_MEMBERS.CONNECTION_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUP_MEMBERS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUP_MEMBERS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_GROUP_MEMBERS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_GROUP_MEMBERS creation'); 

END create_MD_GROUP_MEMBERS;


PROCEDURE create_MD_GROUP_PRIVILEGES
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_GROUP_PRIVILEGES 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE) NOT NULL,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     GROUP_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     PRIVILEGE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     PRIVELEGEOBJECTTYPE VARCHAR2 (255 BYTE)  NOT NULL , 
	     PRIVELEGE_TYPE VARCHAR2 (256 BYTE)  NOT NULL , 
	     GROUP_ID_FK NUMBER  NOT NULL , 
	     PRIVILEGE_ID_FK NUMBER  NOT NULL , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_GROUP_PRIVILEGES IS 'This table stores the privileges granted to a group (or role)'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUP_MEMBERS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GROUP_MEMBERS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_GROUP_PRIVILEGES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_GROUP_PRIVILEGES creation'); 

END create_MD_GROUP_PRIVILEGES;



PROCEDURE create_MD_INDEXES
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_INDEXES 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     TABLE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     COLUMN_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     INDEX_NAME VARCHAR2 (256 BYTE)  NOT NULL ,
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     INDEX_TYPE VARCHAR2 (256 BYTE) ,
	     INDEX_PORTION NUMBER ,
	     IS_DESCENDING CHAR(1 Byte) DEFAULT 'N' NOT NULL, 
	     DETAIL_ORDER NUMBER  NOT NULL ,
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) ,
	     ISCLUSTEREDINDEX CHAR(1 BYTE) DEFAULT 'N' NOT NULL, --Y Yes N No
	     COMMENTS VARCHAR2 (4000 BYTE) ,
	     INDEX_NAMESPACE VARCHAR2 (256 BYTE),
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (4000 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_INDEXES IS 'This table is used to store information about the indexes in a schema'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_INDEXES.INDEX_NAME IS 'Name of the index //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_INDEXES.INDEX_TYPE IS 'Type of the index e.g. PRIMARY' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_INDEXES.NATIVE_SQL IS 'SQL used to create the index at source'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_INDEXES.NATIVE_KEY IS 'A unique identifier for this object at the source'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_INDEXES.ISCLUSTEREDINDEX IS 'Indicates whether the index is Clustered. Values are Y for yes or N for no.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_INDEXES.INDEX_NAMESPACE IS 'Shared Index Namespace(within a Schema) for CONVERTED type'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_INDEXES.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_INDEXES.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_INDEXES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_INDEXES creation'); 

END create_MD_INDEXES;



PROCEDURE create_MD_MIGR_DEPENDENCY
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_MIGR_DEPENDENCY 
	    ( 
	     ID NUMBER  NOT NULL , 
	     CONNECTION_ID_FK NUMBER  NOT NULL , 
	     PARENT_ID NUMBER  NOT NULL , 
	     CHILD_ID NUMBER  NOT NULL , 
	     PARENT_OBJECT_TYPE VARCHAR2 (256 BYTE)  NOT NULL , 
	     CHILD_OBJECT_TYPE VARCHAR2 (256 BYTE)  NOT NULL , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) , 
	     MD_CONNECTIONS_PROJECT_ID_FK NUMBER  NOT NULL 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_MIGR_DEPENDENCY.CONNECTION_ID_FK IS 'The connection that this exists in //PARENTFIELD' 
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_MIGR_DEPENDENCY created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_MIGR_DEPENDENCY creation'); 

END create_MD_MIGR_DEPENDENCY;

PROCEDURE create_MD_MIGR_PARAMETER
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_MIGR_PARAMETER 
	    ( 
	     ID NUMBER  NOT NULL , 
	     CONNECTION_ID_FK NUMBER  NOT NULL , 
	     OBJECT_ID NUMBER  NOT NULL , 
	     OBJECT_TYPE VARCHAR2 (256 BYTE)  NOT NULL , 
	     PARAM_EXISTING NUMBER  NOT NULL , 
	     PARAM_ORDER NUMBER  NOT NULL , 
	     PARAM_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     PARAM_TYPE VARCHAR2 (256 BYTE)  NOT NULL , 
	     PARAM_DATA_TYPE VARCHAR2 (256 BYTE)  NOT NULL , 
	     PERCISION NUMBER , 
	     SCALE NUMBER , 
	     NULLABLE CHAR (1 BYTE)  NOT NULL , 
	     DEFAULT_VALUE VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) , 
	     MD_CONNECTIONS_PROJECT_ID_FK NUMBER  NOT NULL 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_MIGR_PARAMETER.CONNECTION_ID_FK IS 'the connection in which this belongs //PARENTFIELD'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_MIGR_PARAMETER.PARAM_EXISTING IS '1 represents a new parameter for PL/SQL that was not present in the origional. 0 represents a n existing parameter that was present in the origional' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_MIGR_PARAMETER.PARAM_ORDER IS 'IF -1 THEN THIS PARAM IS A RETURN PARAMETER. 1 WILL BE THE FIRST PARAMETER IN THE PARAMETER LIST' 
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_MIGR_PARAMETER created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_MIGR_PARAMETER creation'); 

END create_MD_MIGR_PARAMETER;


PROCEDURE create_MD_MIGR_WEAKDEP
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_MIGR_WEAKDEP 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     OBJECT_ID NUMBER  NOT NULL , 
	     PARENT_ID NUMBER  NOT NULL , 
	     CHILD_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     PARENT_TYPE VARCHAR2 (256 BYTE)  NOT NULL , 
	     CHILD_TYPE VARCHAR2 (256 BYTE)  NOT NULL , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	 
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_MIGR_WEAKDEP.CHILD_NAME IS 'name of the child,  as weak dependencies dont have reference to child id'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_MIGR_WEAKDEP.PARENT_TYPE IS 'MD_<tablename>' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_MIGR_WEAKDEP.CHILD_TYPE IS 'Generic Type (not MD_<tablename>)' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_MIGR_WEAKDEP.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_MIGR_WEAKDEP.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_MIGR_WEAKDEP created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_MIGR_WEAKDEP creation'); 

END create_MD_MIGR_WEAKDEP;

PROCEDURE create_MD_NUMROW$SOURCE
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_NUMROW$SOURCE 
	    ( 
	     NUMROWS NUMBER (10) , 
	     NAME VARCHAR2 (4000 BYTE) , 
	     OBJID NUMBER (10) 
	    ) 
	 ]'); 
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_NUMROW$SOURCE created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_NUMROW$SOURCE creation'); 

END create_MD_NUMROW$SOURCE;

PROCEDURE create_MD_NUMROW$TARGET
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_NUMROW$TARGET 
	    ( 
	     NUMROWS NUMBER (10) , 
	     NAME VARCHAR2 (4000 BYTE) , 
	     OBJID NUMBER (10) 
	    ) 
	 ]'); 
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_NUMROW$TARGET created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_NUMROW$TARGET creation'); 

END create_MD_NUMROW$TARGET;


PROCEDURE create_MD_OTHER_OBJECTS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_OTHER_OBJECTS 
	    ( 
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL , 
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     OBJECT_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) , 
	     SCHEMA_ID_FK NUMBER  NOT NULL 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_OTHER_OBJECTS IS 'For storing objects that don''''t belong anywhere else'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_OTHER_OBJECTS.CONNECTION_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_OTHER_OBJECTS.OBJECT_NAME IS 'Name of this object //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_OTHER_OBJECTS.NATIVE_SQL IS 'The native SQL used to create this object' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_OTHER_OBJECTS.NATIVE_KEY IS 'A key that identifies this object at source.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_OTHER_OBJECTS.SCHEMA_ID_FK IS 'Schema to which this object blongs. //PARENTFIELD'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_OTHER_OBJECTS.ORACLE_NAME IS 'Transformed name of object.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_OTHER_OBJECTS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_OTHER_OBJECTS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_OTHER_OBJECTS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_OTHER_OBJECTS creation'); 

END create_MD_OTHER_OBJECTS;


PROCEDURE create_MD_PACKAGES
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_PACKAGES 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL , 
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     PACKAGE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     SCHEMA_ID_FK NUMBER  NOT NULL , 
	     PACKAGE_HEADER CLOB , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     LANGUAGE VARCHAR2 (40 BYTE)  NOT NULL ,
	     SCHEMA_NAMESPACE VARCHAR2 (256 BYTE),
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_PACKAGES IS 'For storing packages'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PACKAGES.CONNECTION_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PACKAGES.PACKAGE_NAME IS 'Name of this package //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PACKAGES.SCHEMA_ID_FK IS 'the schema in which this package resides //PARENTFIELD'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PACKAGES.NATIVE_SQL IS 'The SQL used to create this package at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PACKAGES.NATIVE_KEY IS 'A unique identifer for this object at source.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PACKAGES.LANGUAGE IS '//PUBLIC'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PACKAGES.ORACLE_NAME IS 'Transformed name of package.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PACKAGES.SCHEMA_NAMESPACE IS 'Shared Schema Namespace for CONVERTED type'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PACKAGES.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PACKAGES.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_PACKAGES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_PACKAGES creation'); 

END create_MD_PACKAGES;


PROCEDURE create_MD_PARTITIONS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_PARTITIONS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     TABLE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     TABLE_ID_FK NUMBER  NOT NULL , 
	     NATIVE_SQL CLOB , 
	     PARTITION_EXPRESSION VARCHAR2 (4000 BYTE) , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_PARTITIONS IS 'Partition information is stored in this table.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PARTITIONS.PROJECT_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PARTITIONS.CONNECTION_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PARTITIONS.TABLE_ID_FK IS 'The table that this partition refers to //PARENTFIELD' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PARTITIONS.NATIVE_SQL IS 'The SQL used to create this partition at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PARTITIONS.PARTITION_EXPRESSION IS 'The partition expression' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PARTITIONS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PARTITIONS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_PARTITIONS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_PARTITIONS creation'); 

END create_MD_PARTITIONS;

PROCEDURE create_MD_PRIVILEGES
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_PRIVILEGES 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE) NOT NULL,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     PRIVILEGE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     PRIVELEGEOBJECTTYPE VARCHAR2 (255 BYTE)  NOT NULL , 
	     PRIVELEGE_TYPE VARCHAR2 (256 BYTE)  NOT NULL , 
	     ADMIN_OPTION CHAR (1 BYTE) , 
	     NATIVE_SQL CLOB  NOT NULL , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_PRIVILEGES IS 'This table stores privilege information'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PRIVILEGES.CONNECTION_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PRIVILEGES.PRIVILEGE_NAME IS 'The name of the privilege //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PRIVILEGES.PRIVELEGEOBJECTTYPE IS 'The type the privelege is on (e.g. INDEX)'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PRIVILEGES.PRIVELEGE_TYPE IS 'e.g.select'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PRIVILEGES.ADMIN_OPTION IS 'Flag to show if this was granted with admin option.  ''Y'' means it was granted with admin option ''N'' means it was NOT granted with admin option.  NULL means not applicable (e.g. not known, not supported by source platform, etc.)' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PRIVILEGES.NATIVE_SQL IS 'The SQL used to create this privilege at source'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PRIVILEGES.NATIVE_KEY IS 'An identifier for this object at source.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PRIVILEGES.ORACLE_NAME IS 'Transformed name of privilege.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PRIVILEGES.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PRIVILEGES.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_PRIVILEGES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_PRIVILEGES creation'); 

END create_MD_PRIVILEGES;


PROCEDURE create_MD_PROJECTS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_PROJECTS 
	    ( 
	     PROJECT_ID NUMBER  NOT NULL , 
	     PROJECT_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_PROJECTS IS 'This is a top level container for a set of connections.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PROJECTS.PROJECT_ID IS 'Primary key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PROJECTS.PROJECT_NAME IS 'Name of the project //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_PROJECTS.COMMENTS IS 'All migrations happen in Projects' 
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_PROJECTS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_PROJECTS creation'); 

END create_MD_PROJECTS;

PROCEDURE create_MD_REGISTRY
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_REGISTRY 
	    ( 
	     OBJECT_TYPE VARCHAR2 (30 BYTE)  NOT NULL , 
	     OBJECT_NAME VARCHAR2 (30 BYTE)  NOT NULL , 
	     DESC_OBJECT_NAME VARCHAR2 (30 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_REGISTRY IS 'Table to store information on the MD_ repository.  This lists the objects to be dropped if you wish to remove the repository'
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_REGISTRY created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_REGISTRY creation'); 

END create_MD_REGISTRY;

PROCEDURE create_MD_REPOVERSIONS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_REPOVERSIONS 
	    (
	     REVISION NUMBER  NOT NULL 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_REPOVERSIONS IS 'This table is used to version this schema for future requirements.'
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_REPOVERSIONS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_REPOVERSIONS creation'); 

END create_MD_REPOVERSIONS;

PROCEDURE create_MD_SCHEMAS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_SCHEMAS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE) NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE), 
	     CHARACTER_SET VARCHAR2 (4000 BYTE) , 
	     VERSION_TAG VARCHAR2 (40 BYTE) , 
	     NATIVE_SQL CLOB , 
	     USER_NAMESPACE VARCHAR2 (256 BYTE),
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_SCHEMAS IS 'This is the holder for schemas'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.CONNECTION_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.CATALOG_NAME IS 'Catalog to which this schema blongs //PARENTFIELD'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.SCHEMA_NAME IS 'Name of the schema //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.TYPE IS 'Type of this schema.  Eaxamples are ''CAPTURED'' OR ''CONVERTED''' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.CHARACTER_SET IS 'The characterset of this schema'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.VERSION_TAG IS 'A version string that can be used for tagging/baseling a schema' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.NATIVE_SQL IS 'The native SQL used to create this schema'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.USER_NAMESPACE IS 'The shared Namespace among Schema objects'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.NATIVE_KEY IS 'A unique identifier that this schema is known as in its source state.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.ORACLE_NAME IS 'Transformed name of schema.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SCHEMAS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_SCHEMAS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_SCHEMAS creation'); 

END create_MD_SCHEMAS;

PROCEDURE create_MD_SEQUENCES
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_SEQUENCES 
	    (
	     -- VERSION_ID NUMBER NOT NULL, 
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE) NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     SEQUENCE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     SEQ_START NUMBER  NOT NULL , 
	     CURRENT_VALUE NUMBER NOT NULL,
	     INCR NUMBER DEFAULT 1  NOT NULL , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) ,
	     SCHEMA_NAMESPACE VARCHAR2 (256 BYTE),
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) DEFAULT '0'  NOT NULL , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) , 
	     MIN NUMBER , 
	     MAX NUMBER , 
	     CACHE NUMBER (1) ,
	     CACHE_SIZE NUMBER(10,0) , 
	     "ORDER" NUMBER (1) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_SEQUENCES IS 'For storing information on sequences.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.SEQUENCE_NAME IS 'Name of this sequence //OBJECTNAME' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.SEQ_START IS 'Starting point of the sequence' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.INCR IS 'Increment value of the sequence' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.NATIVE_SQL IS 'SQL used to create this object at source'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.NATIVE_KEY IS 'Identifier for this object at source.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.SCHEMA_NAMESPACE IS 'Shared Schema Namespace value for CONVERTED TYPE'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.CACHE IS '1=true 0=false'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES."ORDER" IS '1=yes 2=no'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.ORACLE_NAME IS 'Transformed name of sequence.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.CURRENT_VALUE IS 'Current value of sequence'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.CURRENT_VALUE IS 'Size of cache if any' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SEQUENCES.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_SEQUENCES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_SEQUENCES creation'); 

END create_MD_SEQUENCES;

PROCEDURE create_MD_STORED_PROGRAMS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_STORED_PROGRAMS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE) NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     PROGRAM_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     PROGRAMTYPE VARCHAR2 (20 BYTE) ,
	     FUNCTIONTYPE VARCHAR2 (20 BYTE),
	     PACKAGE_ID_FK NUMBER , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     LANGUAGE VARCHAR2 (40 BYTE)  NOT NULL , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     LINECOUNT NUMBER , 
	     PROP001           VARCHAR2 (4000 BYTE) ,
	     PROP002           VARCHAR2 (4000 BYTE) ,
	     SCHEMA_NAMESPACE VARCHAR2 (256 BYTE),
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) , 
	     MD_PACKAGES_PROJECT_ID NUMBER , 
	     MD_PACKAGES_CATALOG_NAME VARCHAR2 (256 BYTE) , 
	     MD_PACKAGES_TYPE VARCHAR2 (32 BYTE),
	     MD_PACKAGES_SCHEMA_NAME VARCHAR2 (256 BYTE) , 
	     MD_PACKAGES_PACKAGE_NAME VARCHAR2 (256 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_STORED_PROGRAMS IS 'Container for stored programs.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.PROJECT_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.SCHEMA_NAME IS 'Schema to which this object belongs.  Note that the PACKAGE_ID_FK (if present also leads us to the relevant schema), however a stored program may or may not belong in a package.  If it is in a package, then the SCHEMA_ID_FK and the SCHEME_ID_FK in the parent package should match //PARENTFIELD' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.PROGRAM_NAME IS 'Name of the stored program //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.PROGRAMTYPE IS 'Java/TSQL/PLSQL, etc.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.PROP001 IS 'From SQLServer - (moved from  Program_subtype) was Additional property: SCALAR FUNCTION TABLE FUNCTION OR INLINED T. FUNCTION' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.PROP002 IS 'Not currently used'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.PACKAGE_ID_FK IS 'The package to which this object belongs'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.NATIVE_SQL IS 'The SQL used to create this object at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.NATIVE_KEY IS 'A unique indetifier for this object at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.LANGUAGE IS '//PUBLIC'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.ORACLE_NAME IS 'Transformed name of stored program.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.SCHEMA_NAMESPACE IS 'Shared Schema Namespace value for CONVERTED type'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_STORED_PROGRAMS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_STORED_PROGRAMS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_STORED_PROGRAMS creation'); 

END create_MD_STORED_PROGRAMS;

PROCEDURE create_MD_SYNONYMS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_SYNONYMS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     SYNONYM_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     SCHEMA_ID_FK NUMBER, 
	     SYNONYM_FOR_OBJECT_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     FOR_OBJECT_TYPE VARCHAR2 (256 BYTE) , 
	     PRIVATE_VISIBILITY CHAR (1 BYTE) , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     COMMENTS VARCHAR2 (4000 BYTE) ,
	     SCHEMA_NAMESPACE VARCHAR2 (256 BYTE),
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_SYNONYMS IS 'For storing synonym information.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.SYNONYM_NAME IS 'Synonym name //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.SCHEMA_ID_FK IS 'The schema to which this object belongs (might be null so removing not null)//PARENTFIELD'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.SYNONYM_FOR_OBJECT_NAME IS 'What object this is a synonym for' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.FOR_OBJECT_TYPE IS 'The type this is a synonym for (e.g. INDEX)'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.PRIVATE_VISIBILITY IS 'Visibility - Private or Public.  If Private_visibility = ''Y'' means this is a private synonym.  Anything else means it is a public synonym' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.NATIVE_SQL IS 'The SQL used to create this object at source'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.NATIVE_KEY IS 'An identifier for this object at source.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.ORACLE_NAME IS 'Transformed name of synonym.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.SCHEMA_NAMESPACE IS 'Shared Schema Namespace value for CONVERTED TYPE'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_SYNONYMS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_SYNONYMS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_SYNONYMS creation'); 

END create_MD_SYNONYMS;

PROCEDURE create_MD_TABLES
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_TABLES 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE) NOT NULL,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     TABLE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     NUM_ROWS NUMBER , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) ,
	     ISINDEXTABLE CHAR(1 BYTE) DEFAULT 'N' NOT NULL, --Y Yes N No
	     QUALIFIED_NATIVE_NAME VARCHAR2 (4000 BYTE)  NOT NULL , 
	     SCHEMA_NAMESPACE VARCHAR2 (256 BYTE),
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_TABLES IS 'Table used to store information about tables.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.PROJECT_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.SCHEMA_NAME IS 'Schema in which this table resides //PARENTFIELD'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.TABLE_NAME IS 'Name of the table //OBJECTNAME' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.NUM_ROWS IS 'In captured model, this will be the rows we have identified that come from the source db.  In the the target model, this will be the rows which we have moved and confirmed to move.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.NATIVE_SQL IS 'SQL Used to create this table at soruce'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.NATIVE_KEY IS 'Unique identifier for this table at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.ISINDEXTABLE IS 'Populated by trigger MD_INDEXES_ISCLUSTERED.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.ORACLE_NAME IS 'Transformed name of table.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.SCHEMA_NAMESPACE IS 'Shared Schema Namespace for CONVERTED type'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLES.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_TABLES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_TABLES creation'); 

END create_MD_TABLES;

PROCEDURE create_MD_TABLESPACES
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_TABLESPACES 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     TABLESPACE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_TABLESPACES IS 'For storing information about tablespaces.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLESPACES.PROJECT_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLESPACES.CONNECTION_ID IS 'Schema to which this tablespace belongs //PARENTFIELD' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLESPACES.TABLESPACE_NAME IS 'Name of the table space //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLESPACES.NATIVE_SQL IS 'The SQL used to create this tablespace' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLESPACES.NATIVE_KEY IS 'A unique identifier for this object at source'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLESPACES.ORACLE_NAME IS 'Transformed name of table space.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLESPACES.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TABLESPACES.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_TABLESPACES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_TABLESPACES creation'); 

END create_MD_TABLESPACES;

PROCEDURE create_MD_TRIGGERS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_TRIGGERS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     TABLE_OR_VIEW_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     TRIGGER_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     TRIGGER_ON_FLAG CHAR (1 BYTE)  NOT NULL , 
	     TRIGGER_TIMING VARCHAR2 (4000 BYTE) , 
	     TRIGGER_OPERATION VARCHAR2 (4000 BYTE) , 
	     TRIGGER_EVENT VARCHAR2 (4000 BYTE) , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     LANGUAGE VARCHAR2 (40 BYTE)  NOT NULL , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     LINECOUNT NUMBER ,
	     TRIGGER_NAMESPACE VARCHAR2 (256 BYTE),
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_TRIGGERS IS 'For storing information about triggers.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.PROJECT_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.CONNECTION_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.CATALOG_NAME IS 'Name of the trigger //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.SCHEMA_NAME IS 'Name of the trigger //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.TABLE_OR_VIEW_NAME IS 'Table on which this trigger fires' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.TRIGGER_NAME IS 'Name of the trigger //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.TRIGGER_ON_FLAG IS 'Flag to show iif the trigger is on a table or a view.  If it is a table this should be ''T''. If it is on a view this should be ''V''' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.TRIGGER_TIMING IS 'before, after ,etc.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.TRIGGER_OPERATION IS 'insert, delete, etc.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.TRIGGER_EVENT IS 'The actual trigger that gets fired '
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.NATIVE_SQL IS 'The full definition ' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.NATIVE_KEY IS 'UInique identifer for this object at source'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.LANGUAGE IS '//PUBLIC' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.ORACLE_NAME IS 'Transformed name of trigger.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.TRIGGER_NAMESPACE IS 'Shared Trigger Namespace(within a Schema) for CONVERTED type'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_TRIGGERS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_TRIGGERS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_TRIGGERS creation'); 

END create_MD_TRIGGERS;

PROCEDURE create_MD_USERS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_USERS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     USERNAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     PASSWORD VARCHAR2 (1000 BYTE) , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_USERS IS 'User information.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USERS.PROJECT_ID IS 'Shema in which this object belongs //PARENTFIELD' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USERS.USERNAME IS 'Username for login //OBJECTNAME' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USERS.PASSWORD IS 'Password for login'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USERS.NATIVE_SQL IS 'SQL Used to create this object at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USERS.NATIVE_KEY IS 'Unique identifier for this object at source.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USERS.ORACLE_NAME IS 'Transformed name of user.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USERS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USERS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_USERS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_USERS creation'); 

END create_MD_USERS;

PROCEDURE create_MD_USER_DEFINED_DATA_TYPES
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_USER_DEFINED_DATA_TYPES 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     DATA_TYPE_NAME VARCHAR2 (256 BYTE)  NOT NULL ,
	     ORACLE_NAME VARCHAR2 (256 BYTE),
	     DEFINITION VARCHAR2 (4000 BYTE)  NOT NULL , 
	     NATIVE_SQL CLOB  NOT NULL , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     COMMENTS VARCHAR2 (4000 BYTE) ,
	     SCHEMA_NAMESPACE VARCHAR2 (256 BYTE),
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_USER_DEFINED_DATA_TYPES IS 'For storing information on user defined data types.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_DEFINED_DATA_TYPES.CONNECTION_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_DEFINED_DATA_TYPES.DATA_TYPE_NAME IS 'The name of the data type //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_DEFINED_DATA_TYPES.DEFINITION IS 'The definition of the data type'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_DEFINED_DATA_TYPES.NATIVE_SQL IS 'The native SQL used to create this object'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_DEFINED_DATA_TYPES.NATIVE_KEY IS 'An unique identifier for this object at source.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_DEFINED_DATA_TYPES.SCHEMA_NAMESPACE IS 'Shared Schema namespace value for CONVERTED type.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_DEFINED_DATA_TYPES.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_DEFINED_DATA_TYPES.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_USER_DEFINED_DATA_TYPES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_USER_DEFINED_DATA_TYPES creation'); 

END create_MD_USER_DEFINED_DATA_TYPES;

PROCEDURE create_MD_USER_PRIVILEGES
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_USER_PRIVILEGES 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     USERNAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     PRIVILEGE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     PRIVELEGEOBJECTTYPE VARCHAR2 (255 BYTE)  NOT NULL , 
	     PRIVELEGE_TYPE VARCHAR2 (256 BYTE)  NOT NULL , 
	     USER_ID_FK NUMBER  NOT NULL , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UDPATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_USER_PRIVILEGES IS 'This table stores privileges granted to individual users'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_PRIVILEGES.CONNECTION_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_PRIVILEGES.USER_ID_FK IS 'User'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_PRIVILEGES.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_USER_PRIVILEGES.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_USER_PRIVILEGES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_USER_PRIVILEGES creation'); 

END create_MD_USER_PRIVILEGES;

PROCEDURE create_MD_VIEWS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_VIEWS 
	    (
	     -- VERSION_ID NUMBER NOT NULL,
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     "TYPE" VARCHAR2 (32 BYTE)  NOT NULL ,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     VIEW_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     ORACLE_NAME VARCHAR2 (256 BYTE) , 
	     NATIVE_SQL CLOB , 
	     NATIVE_KEY VARCHAR2 (4000 BYTE) , 
	     LANGUAGE VARCHAR2 (40 BYTE)  NOT NULL , 
	     COMMENTS VARCHAR2 (4000 BYTE) , 
	     LINECOUNT NUMBER ,
	     SCHEMA_NAMESPACE VARCHAR2 (256 BYTE),
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_VIEWS IS 'For storing information on views.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_VIEWS.VIEW_NAME IS 'The name of the view //OBJECTNAME'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_VIEWS.NATIVE_SQL IS 'The SQL used to create this object at source' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_VIEWS.NATIVE_KEY IS 'An identifier for this object at source.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_VIEWS.LANGUAGE IS '//PUBLIC'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_VIEWS.ORACLE_NAME IS 'Transformed name of view.' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_VIEWS.SCHEMA_NAMESPACE IS 'Shared Schema Namespace value.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_VIEWS.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_VIEWS.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_VIEWS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_VIEWS creation'); 

END create_MD_VIEWS;

PROCEDURE create_MD_RESERVED_WORDS
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_RESERVED_WORDS 
	    ( 
	     KEYWORD VARCHAR2 (40 BYTE)  NOT NULL 
	    ) 
	 ]'); 
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_RESERVED_WORDS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_RESERVED_WORDS creation'); 

END create_MD_RESERVED_WORDS;

PROCEDURE create_MD_MIGRLOG
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_MIGRLOG 
	    ( 
	     ID NUMBER  NOT NULL , 
	     PARENT_LOG_ID NUMBER , 
	     LOG_DATE TIMESTAMP  NOT NULL , 
	     SEVERITY NUMBER (4)  NOT NULL , 
	     LOGTEXT VARCHAR2 (4000 BYTE) , 
	     PHASE VARCHAR2 (100 BYTE) , 
	     REF_OBJECT_ID NUMBER , 
	     REF_OBJECT_TYPE VARCHAR2 (4000 BYTE) , 
	     CONNECTION_ID_FK NUMBER 
	    ) 
	 ]'); 
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_MIGRLOG created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_MIGRLOG creation'); 

END create_MD_MIGRLOG;

PROCEDURE create_MD_DATATYPE_TRANSFORM_MAP
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_DATATYPE_TRANSFORM_MAP 
	    ( 
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     MAP_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 

	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_DATATYPE_TRANSFORM_MAP IS 'Table for storing data type maps.  A map is simply a collection of rules'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_MAP.PROJECT_ID IS '//PARENTFIELD' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_MAP.CONNECTION_ID IS 'Primary Key' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_MAP.MAP_NAME IS 'A name for the map' 
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_DATATYPE_TRANSFORM_MAP created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_DATATYPE_TRANSFORM_MAP creation'); 

END create_MD_DATATYPE_TRANSFORM_MAP;

PROCEDURE create_MD_DATATYPE_TRANSFORM_RULE
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_DATATYPE_TRANSFORM_RULE
	    ( 
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     MAP_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     SOURCE_DATA_TYPE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     SOURCE_PRECISION NUMBER , 
	     SOURCE_SCALE NUMBER , 
	     TARGET_DATA_TYPE_NAME VARCHAR2 (256 BYTE)  NOT NULL , 
	     TARGET_PRECISION NUMBER , 
	     TARGET_SCALE NUMBER , 
	     SECURITY_GROUP_ID NUMBER DEFAULT 0  NOT NULL , 
	     CREATED_ON DATE DEFAULT sysdate  NOT NULL , 
	     CREATED_BY VARCHAR2 (255 BYTE) , 
	     LAST_UPDATED_ON DATE , 
	     LAST_UPDATED_BY VARCHAR2 (255 BYTE) 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_RULE.PROJECT_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_RULE.CONNECTION_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_RULE.MAP_NAME IS 'The map to which this rule belongs //PARENTFIELD'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_RULE.SOURCE_DATA_TYPE_NAME IS 'Source data type' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_RULE.SOURCE_PRECISION IS 'Precison to match' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_RULE.SOURCE_SCALE IS 'scale to match'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_RULE.TARGET_DATA_TYPE_NAME IS 'data type name to transform to' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_RULE.TARGET_PRECISION IS 'precision to map to ' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_DATATYPE_TRANSFORM_RULE.TARGET_SCALE IS 'scale to map to' 
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_DATATYPE_TRANSFORM_RULE created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_DATATYPE_TRANSFORM_RULE creation'); 

END create_MD_DATATYPE_TRANSFORM_RULE;

PROCEDURE create_MD_GENERATION_ORDER
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_GENERATION_ORDER 
	    ( 
	     ID NUMBER  NOT NULL , 
	     CONNECTION_ID_FK NUMBER  NOT NULL , 
	     OBJECT_ID NUMBER  NOT NULL , 
	     OBJECT_TYPE VARCHAR2 (4000 BYTE)  NOT NULL , 
	     GENERATION_ORDER NUMBER  NOT NULL , 
	     MD_CONNECTIONS_PROJECT_ID_FK NUMBER  NOT NULL 
	    ) 
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GENERATION_ORDER.CONNECTION_ID_FK IS '//PARENTFIELD' 
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_GENERATION_ORDER created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_GENERATION_ORDER creation'); 

END create_MD_GENERATION_ORDER;

PROCEDURE create_MD_GENERATE
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_GENERATE 
	    ( 
	     PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL, 
	     SCHEMA_NAME VARCHAR2 (256 BYTE)  NOT NULL,
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     GENERATE_VERSION NUMBER  NOT NULL ,
	     OBJECT_TYPE VARCHAR2 (256 BYTE)  NOT NULL, 
	     OBJECT_NAME VARCHAR2 (256 BYTE)  NOT NULL,  
	     SQL_SCRIPT CLOB, 
	     COMMENTS VARCHAR2(4000 BYTE), 
	     CREATED_ON TIMESTAMP DEFAULT CURRENT_TIMESTAMP(3), 
	     CREATED_BY VARCHAR2(255 BYTE)
	    )
	 ]'); 
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_GENERATE IS 'Table used to store information about Migration Code generation.'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GENERATE.PROJECT_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GENERATE.CATALOG_NAME IS 'Catalog in which this schema resides //PARENTFIELD'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GENERATE.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GENERATE.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GENERATE.GENERATE_VERSION IS 'Generated version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GENERATE.OBJECT_NAME IS 'Name of the Object //OBJECTNAME' 
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GENERATE.OBJECT_TYPE IS 'Name of the Object //OBJECTYPE'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_GENERATE.SQL_SCRIPT IS 'SQL Used to create this table at soruce'
	 ]');
	 
    DBMS_OUTPUT.PUT_LINE('Table MD_GENERATE created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_GENERATE creation'); 

END create_MD_GENERATE;

PROCEDURE create_MD_REPORT
IS
BEGIN
	EXECUTE IMMEDIATE (q'[
	  CREATE TABLE MD_REPORT 
	   ( PROJECT_ID NUMBER  NOT NULL , 
	     CONNECTION_ID NUMBER  NOT NULL , 
	     CATALOG_NAME VARCHAR2 (256 BYTE)  NOT NULL, 
	     CAPT_VERSION NUMBER  NOT NULL ,
	     CONV_VERSION NUMBER  NOT NULL ,
	     GENERATE_VERSION NUMBER  NOT NULL ,
	     OPERATION_ID NUMBER  NOT NULL , 
	     SOURCE_SCHEMA_NAME VARCHAR2 (256 BYTE),
	     SOURCE_TABLE_NAME VARCHAR2 (256 BYTE),
         SOURCE_TABLE_ROWS NUMBER,
         TARGET_SCHEMA_NAME VARCHAR2 (256 BYTE),
	     TARGET_TABLE_NAME VARCHAR2 (256 BYTE),
         TARGET_TABLE_ROWS_LOADED NUMBER,
         LOGFILE_TABLE VARCHAR2 (256 BYTE),
         BADFILE_TABLE VARCHAR2 (256 BYTE),
         STATUS VARCHAR2 (256 BYTE),
         ERROR_MESSAGES VARCHAR2 (2000 BYTE),
	     COMMENTS VARCHAR2(4000 BYTE), 
	     START_TIME TIMESTAMP(6) WITH TIME ZONE,
	     END_TIME TIMESTAMP(6) WITH TIME ZONE,
	     CREATED_BY VARCHAR2(255 BYTE)
	    )
	 ]'); 	
	
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON TABLE MD_REPORT IS 'Table used to store information about Migration Report.'
	 ]');
	 EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.PROJECT_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.CATALOG_NAME IS 'Catalog in which this schema resides //PARENTFIELD'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.CAPT_VERSION IS 'Captured version: To support multiple versions of Capture within the same Project'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.CONV_VERSION IS 'Converted version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	 EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.GENERATE_VERSION IS 'Generated version: To support multiple versions of Convert based on particular version of Capture within the same Project '
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.OPERATION_ID IS 'Primary Key'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.SOURCE_SCHEMA_NAME IS 'Source schema name'
	 ]'); 
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.SOURCE_TABLE_NAME IS 'Source table name'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.SOURCE_TABLE_ROWS IS 'Source table rows'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.TARGET_SCHEMA_NAME IS 'Target schema name'
	 ]'); 
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.TARGET_TABLE_NAME IS 'Target table name'
	 ]');
	EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.TARGET_TABLE_ROWS_LOADED IS 'Target table rows loaded'
	 ]');
	 EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.LOGFILE_TABLE IS 'Data load log file name from dba_load_operations table'
	 ]');
	 EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.BADFILE_TABLE IS 'Rejected rows storage table badfile table name from dba_load_operations table'
	 ]');
	 EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.STATUS IS 'Data load status : COMPLETED, PROGRESS, FAILED'
	 ]');
	 EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.ERROR_MESSAGES IS 'Stores the Error Message for failed operations'
	 ]');
	 EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.START_TIME IS 'Data loading start time'
	 ]');
	 EXECUTE IMMEDIATE (q'[ 
	  COMMENT ON COLUMN MD_REPORT.END_TIME IS 'Data loading end time'
	 ]');
	
    DBMS_OUTPUT.PUT_LINE('Table MD_REPORT created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_REPORT creation'); 

END create_MD_REPORT;


PROCEDURE insertOracleReservedWords
IS
BEGIN

	/* Add reserved words to MD_RESERVED_WORDS so a clash with SHARE will mean it renamed to SHARE_ */
	
	EXECUTE IMMEDIATE (q'[ 
	  INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('SHARE')
	 ]');
	 
	EXECUTE IMMEDIATE (q'[                          
 		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('LEVEL')
    ]');
 	
 	EXECUTE IMMEDIATE (q'[                           
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('RAW')
    ]');
	
	EXECUTE IMMEDIATE (q'[                             
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('DROP')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('BETWEEN')
    ]');
	
	EXECUTE IMMEDIATE (q'[                         
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('FROM')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('DESC')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('OPTION')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('PRIOR')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('LONG')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('THEN')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('DEFAULT')
    ]'); 
	
	EXECUTE IMMEDIATE (q'[                        
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ALTER')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('IS')
    ]');
	
	EXECUTE IMMEDIATE (q'[                              
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('INTO')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('MINUS')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('INTEGER')
    ]');
	
	EXECUTE IMMEDIATE (q'[                        
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('NUMBER')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('GRANT')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('IDENTIFIED')
    ]');
	
	EXECUTE IMMEDIATE (q'[                      
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ALL')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('TO')
    ]');
	
	EXECUTE IMMEDIATE (q'[                              
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ORDER')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
		INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ON')
    ]');
	
	EXECUTE IMMEDIATE (q'[                             
	      INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('FLOAT')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('DATE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('HAVING')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('CLUSTER')
    ]');
	
	EXECUTE IMMEDIATE (q'[                         
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('NOWAIT')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('RESOURCE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                        
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ANY')
    ]');
	
	EXECUTE IMMEDIATE (q'[                             
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('TABLE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('INDEX')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('FOR')
    ]');
	
	EXECUTE IMMEDIATE (q'[                             
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('UPDATE')
    ]'); 
	
	EXECUTE IMMEDIATE (q'[                         
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('WHERE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('CHECK')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('SMALLINT')
    ]');
	
	EXECUTE IMMEDIATE (q'[                        
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('WITH')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('DELETE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('BY')
    ]');
	
	EXECUTE IMMEDIATE (q'[                             
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ASC')
    ]');
	
	EXECUTE IMMEDIATE (q'[                             
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('REVOKE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('LIKE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('SIZE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('RENAME')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('NOCOMPRESS')
    ]');
	
	EXECUTE IMMEDIATE (q'[                      
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('NULL')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('GROUP')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('VALUES')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('AS')
    ]');
	
	EXECUTE IMMEDIATE (q'[                              
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('IN')
    ]');
	
	EXECUTE IMMEDIATE (q'[                              
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('VIEW')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('EXCLUSIVE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                       
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('COMPRESS')
    ]');
	
	EXECUTE IMMEDIATE (q'[                        
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('SYNONYM')
    ]');
	
	EXECUTE IMMEDIATE (q'[                         
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('SELECT')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('INSERT')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('EXISTS')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('NOT')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('TRIGGER')
    ]');
	
	EXECUTE IMMEDIATE (q'[                         
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ELSE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('CREATE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('INTERSECT')
    ]');
	
	EXECUTE IMMEDIATE (q'[                       
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('PCTFREE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                         
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('DISTINCT')
    ]');
	
	EXECUTE IMMEDIATE (q'[                        
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('CONNECT')
    ]');
	
	EXECUTE IMMEDIATE (q'[                         
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('SET')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('MODE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('OF')
    ]');
	
	EXECUTE IMMEDIATE (q'[                              
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('UNIQUE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('VARCHAR2')
    ]');
	
	EXECUTE IMMEDIATE (q'[                        
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('VARCHAR')
    ]'); 
	
	EXECUTE IMMEDIATE (q'[                        
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('LOCK')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('OR')
    ]');
	
	EXECUTE IMMEDIATE (q'[                              
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('CHAR')
    ]');
	
	EXECUTE IMMEDIATE (q'[                            
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('DECIMAL')
    ]');
	
	EXECUTE IMMEDIATE (q'[                         
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('UNION')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('PUBLIC')
    ]');
	
	EXECUTE IMMEDIATE (q'[                          
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('AND')
    ]');
	
	EXECUTE IMMEDIATE (q'[                             
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('START')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('USER')
    ]');
	
	EXECUTE IMMEDIATE (q'[                             
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('VALIDATE')
    ]');
	
	EXECUTE IMMEDIATE (q'[                           
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('COMMENT')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('UID')
    ]');
	
	--INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('YEAR');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ROWNUM')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('AUDIT')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ACCESS')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('FILE')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('OFFLINE')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('SESSION')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('PRIVILEGES')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ADD')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('COLUMN')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('CURRENT')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('IMMEDIATE')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('INCREMENT')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('INITIAL')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('MAXEXTENTS')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('MLSLABEL')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('MODIFY')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('NOAUDIT')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ONLINE')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ROW')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ROWID')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('ROWS')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('SUCCESSFUL')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('SYSDATE')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('WHENEVER')
    ]');
	
	EXECUTE IMMEDIATE (q'[ 
	    INSERT INTO MD_RESERVED_WORDS(KEYWORD) VALUES ('COMMIT')
    ]');

	COMMIT;

END insertOracleReservedWords;

PROCEDURE createRepositoryTables
IS
BEGIN
	 create_MD_APPLICATIONFILES;
	 create_MD_APPLICATIONS;
	 create_MD_CATALOGS;
	 create_MD_COLUMNS;
	 create_MD_CONNECTIONS;
	 create_MD_CONNECTION_LINKS;
	 create_MD_CONSTRAINTS;
	 create_MD_FILE_ARTIFACTS;
	 create_MD_GROUPS;
	 create_MD_GROUP_MEMBERS;
	 create_MD_GROUP_PRIVILEGES;
	 create_MD_INDEXES;
	 create_MD_MIGR_DEPENDENCY;
	 create_MD_MIGR_PARAMETER;
	 create_MD_MIGR_WEAKDEP;
	 create_MD_NUMROW$SOURCE;
	 create_MD_NUMROW$TARGET;
	 create_MD_OTHER_OBJECTS;
	 create_MD_PACKAGES;
	 create_MD_PARTITIONS;
	 create_MD_PRIVILEGES;
	 create_MD_PROJECTS;
	 create_MD_REGISTRY;
	 create_MD_REPOVERSIONS;
	 create_MD_SCHEMAS;
	 create_MD_SEQUENCES;
	 create_MD_STORED_PROGRAMS;
	 create_MD_SYNONYMS;
	 create_MD_TABLES;
	 create_MD_TABLESPACES;
	 create_MD_TRIGGERS;
	 create_MD_USERS;
	 create_MD_USER_DEFINED_DATA_TYPES;
	 create_MD_USER_PRIVILEGES;
	 create_MD_VIEWS;
	 create_MD_RESERVED_WORDS;
	 create_MD_MIGRLOG;
	 create_MD_DATATYPE_TRANSFORM_MAP;
	 create_MD_DATATYPE_TRANSFORM_RULE;
	 create_MD_GENERATION_ORDER;
	 create_MD_GENERATE;
	 create_MD_REPORT;
	 
	 insertOracleReservedWords;
 
END createRepositoryTables;

PROCEDURE dropMigrationTables
IS
	sequence_not_found EXCEPTION;
	packageexists number;
	pragma exception_init(sequence_not_found, -2289);
	
	-- CURSORS --
	CURSOR drop_table IS SELECT 'DROP TABLE '||table_name||' CASCADE CONSTRAINTS' query_item FROM USER_TABLES 
	WHERE table_name IN  
	(
	'MD_APPLICATIONFILES', 
	'MD_APPLICATIONS', 
	'MD_CATALOGS',
	'MD_COLUMNS',
	'MD_CONNECTION_LINKS',
	'MD_CONNECTIONS',
	'MD_CONSTRAINT_DETAILS',
	'MD_CONSTRAINTS',
	'MD_FILE_ARTIFACTS',
	'MD_GENERATE',
	'MD_GROUP_MEMBERS',
	'MD_GROUP_PRIVILEGES',
	'MD_GROUPS',
	'MD_INDEX_DETAILS',
	'MD_INDEXES',
	'MD_MIGR_DEPENDENCY',
	'MD_MIGR_PARAMETER',
	'MD_MIGR_WEAKDEP',
	'MD_NUMROW$SOURCE',
	'MD_NUMROW$TARGET',
	'MD_OTHER_OBJECTS',
	'MD_PACKAGES',
	'MD_PARTITIONS',
	'MD_PRIVILEGES',
	'MD_PROJECTS',
	'MD_REGISTRY',
	'MD_REPOVERSIONS',
	'MD_SCHEMAS',
	'MD_SEQUENCES',
	'MD_STORED_PROGRAMS',
	'MD_SYNONYMS',
	'MD_TABLES',
	'MD_TABLESPACES',
	'MD_TRIGGERS',
	'MD_USER_DEFINED_DATA_TYPES',
	'MD_USER_PRIVILEGES',
	'MD_USERS',
	'MD_VIEWS',
	'MD_DATATYPE_TRANSFORM_MAP',
	'MD_DATATYPE_TRANSFORM_RULE',
	'MD_GENERATION_ORDER',
	'MD_RESERVED_WORDS',
	'MD_MIGRLOG',
	'MD_REPORT'
	);

BEGIN
FOR item IN drop_table
  LOOP
    EXECUTE IMMEDIATE item.query_item;
    DBMS_OUTPUT.PUT_LINE('Table ' || REGEXP_REPLACE(item.query_item, '^DROP\s+TABLE\s+((\w|\W)+)\s+CASCADE\s+CONSTRAINTS$', '\1') ||' dropped.');
  END LOOP;
  
  EXECUTE IMMEDIATE 'DROP SEQUENCE DBOBJECTID_SEQUENCE'; 
  EXCEPTION
    WHEN sequence_not_found THEN NULL;

END dropMigrationTables;


PROCEDURE dropMigrationPackages
IS
	-- CURSORS --
	CURSOR drop_package IS SELECT 'DROP PACKAGE '||object_name query_item from USER_OBJECTS where OBJECT_TYPE = 'PACKAGE' and 
	OBJECT_NAME IN  
	(
    'MIGRATION_REPOSITORY_QUERY',
	'MIGRATION_SESSIONSTATE',
	'MIGRATION_TRANSFORMER',
	'REDSHIFT_CAPTURE',
	'MIGRATION_CONVERT',
	'MIGRATION_GENERATOR',
	'REDSHIFT_DEPLOY'
    );

BEGIN
FOR item IN drop_package
  LOOP
    EXECUTE IMMEDIATE item.query_item;
    DBMS_OUTPUT.PUT_LINE('Package ' || REGEXP_REPLACE(item.query_item, '^DROP\s+PACKAGE\s+((\w|\W)+)', '\1') || ' dropped.');
  END LOOP;
  
END dropMigrationPackages;

PROCEDURE dropRepositoryTables
IS
BEGIN

  dropMigrationTables;
 -- dropMigrationPackages;

END dropRepositoryTables;


PROCEDURE truncateRepositoryTables
IS
	-- CURSORS --
	CURSOR truncate_table IS SELECT 'TRUNCATE TABLE '||table_name query_item FROM USER_TABLES 
	WHERE table_name IN  
	(
	'MD_APPLICATIONFILES', 
	'MD_APPLICATIONS', 
	'MD_CATALOGS',
	'MD_COLUMNS',
	'MD_CONNECTION_LINKS',
	'MD_CONNECTIONS',
	'MD_CONSTRAINT_DETAILS',
	'MD_CONSTRAINTS',
	'MD_FILE_ARTIFACTS',
	'MD_GENERATE',
	'MD_GROUP_MEMBERS',
	'MD_GROUP_PRIVILEGES',
	'MD_GROUPS',
	'MD_INDEX_DETAILS',
	'MD_INDEXES',
	'MD_MIGR_DEPENDENCY',
	'MD_MIGR_PARAMETER',
	'MD_MIGR_WEAKDEP',
	'MD_NUMROW$SOURCE',
	'MD_NUMROW$TARGET',
	'MD_OTHER_OBJECTS',
	'MD_PACKAGES',
	'MD_PARTITIONS',
	'MD_PRIVILEGES',
	'MD_PROJECTS',
	'MD_REGISTRY',
	'MD_REPOVERSIONS',
	'MD_SCHEMAS',
	'MD_SEQUENCES',
	'MD_STORED_PROGRAMS',
	'MD_SYNONYMS',
	'MD_TABLES',
	'MD_TABLESPACES',
	'MD_TRIGGERS',
	'MD_USER_DEFINED_DATA_TYPES',
	'MD_USER_PRIVILEGES',
	'MD_USERS',
	'MD_VIEWS',
	'MD_DATATYPE_TRANSFORM_MAP',
	'MD_DATATYPE_TRANSFORM_RULE',
	'MD_GENERATION_ORDER',
--	'MD_RESERVED_WORDS',
	'MD_MIGRLOG',
	'MD_REPORT'
	);

BEGIN
FOR item IN truncate_table
  LOOP
    EXECUTE IMMEDIATE item.query_item;
    DBMS_OUTPUT.PUT_LINE('TRUNCATE TABLE ' || REGEXP_REPLACE(item.query_item, '^TRUNCATE\s+TABLE\s+((\w|\W)+)', '\1'));
  END LOOP;
  
  COMMIT;
  
END truncateRepositoryTables;


PROCEDURE create_MD_STAGE_PG8_DATABASES
IS
BEGIN

	EXECUTE IMMEDIATE (q'[
		CREATE TABLE MD_STAGE_PG8_DATABASES
		(   
		    SERVER_ADDRESS VARCHAR2(255),
		    SERVER_PORT VARCHAR2(255),
		    CATALOG_NAME VARCHAR2(255),
		    SCHEMA_NAME VARCHAR2(255),
		    SCHEMA_OWNER VARCHAR2(255)
		)	
	]');
	
	DBMS_OUTPUT.PUT_LINE('Table MD_STAGE_PG8_DATABASES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_STAGE_PG8_DATABASES creation'); 
	
END create_MD_STAGE_PG8_DATABASES;

PROCEDURE create_MD_STAGE_PG8_TABLES
IS
BEGIN

	EXECUTE IMMEDIATE (q'[
		CREATE TABLE MD_STAGE_PG8_TABLES 
		(   
		    CATALOG_NAME VARCHAR2(255),
		    SCHEMA_NAME VARCHAR2(255),
		    TABLE_NAME VARCHAR2(255),
		    NUM_ROWS_IN_TABLE  NUMBER(10,0)
		)
	]');

	DBMS_OUTPUT.PUT_LINE('Table MD_STAGE_PG8_TABLES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_STAGE_PG8_TABLES creation'); 
	
END create_MD_STAGE_PG8_TABLES;


PROCEDURE create_MD_STAGE_PG8_COLUMNS
IS
BEGIN

	EXECUTE IMMEDIATE (q'[
		CREATE TABLE MD_STAGE_PG8_COLUMNS 
		 (	
			CATALOG_NAME VARCHAR2(255), 
			SCHEMA_NAME VARCHAR2(255), 
			TABLE_NAME VARCHAR2(255), 
			COLUMN_NAME VARCHAR2(255), 
			ORDINAL_POSITION NUMBER(10,0), 
			COLUMN_DEFAULT VARCHAR2(255), 
			DATA_TYPE VARCHAR2(255), 
			CHARACTER_MAXIMUM_LENGTH NUMBER(10,0), 
			CHARACTER_OCTET_LENGTH NUMBER(10,0), 
			NUMERIC_PRECISION NUMBER(10,0), 
			NUMERIC_PRECISION_RADIX NUMBER(10,0), 
			NUMERIC_SCALE NUMBER(10,0), 
			DATETIME_PRECISION NUMBER(10,0), 
			INTERVAL_PRECISION NUMBER(10,0), 
			CHARACTER_SET_CATALOG VARCHAR2(255), 
			CHARACTER_SET_SCHEMA VARCHAR2(255), 
			CHARACTER_SET_NAME VARCHAR2(255), 
			DOMAIN_CATALOG VARCHAR2(255), 
			DOMAIN_SCHEMA VARCHAR2(255), 
			DOMAIN_NAME VARCHAR2(255), 
			UDT_CATALOG VARCHAR2(255), 
			UDT_SCHEMA VARCHAR2(255), 
			UDT_NAME VARCHAR2(255), 
			SCOPE_CATALOG VARCHAR2(255), 
			SCOPE_SCHEMA VARCHAR2(255), 
			SCOPE_NAME VARCHAR2(255), 
			MAXIMUM_CARDINALITY NUMBER(10,0), 
			DTD_IDENTIFIER VARCHAR2(255), 
			IDENTITY_GENERATION VARCHAR2(255), 
			IDENTITY_START VARCHAR2(255), 
			IDENTITY_INCREMENT VARCHAR2(255), 
			IDENTITY_MAXIMUM VARCHAR2(255), 
			IDENTITY_MINIMUM VARCHAR2(255), 
			IS_GENERATED VARCHAR2(255), 
			GENERATION_EXPRESSION VARCHAR2(255)
		 )
	]');

	DBMS_OUTPUT.PUT_LINE('Table MD_STAGE_PG8_COLUMNS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_STAGE_PG8_COLUMNS creation'); 
	
END create_MD_STAGE_PG8_COLUMNS;

PROCEDURE create_MD_STAGE_PG8_CONSTRAINTS
IS
BEGIN

	EXECUTE IMMEDIATE (q'[
		CREATE TABLE MD_STAGE_PG8_CONSTRAINTS 
		(   
		    CATALOG_NAME VARCHAR2(255),
			SCHEMA_NAME VARCHAR2(255),
			CONSTRAINT_NAME VARCHAR2(255),
			CONSTRAINT_TYPE VARCHAR2(255),
			TABLE_NAME VARCHAR2(255),
			COLUMN_NAME VARCHAR2(255),
			ORDINAL_POSITION NUMBER(10,0),
			-- IS_DEFERRABLE VARCHAR2(255),
			-- INITIALLY_DEFERRED VARCHAR2(255),
			-- MATCH_TYPE VARCHAR2(255),
			-- ON_UPDATE VARCHAR2(255),
			-- ON_DELETE VARCHAR2(255),
			REFCATALOG_NAME VARCHAR2(255),
			REFSCHEMA_NAME VARCHAR2(255),
			REFTABLE_NAME VARCHAR2(255),
			REFCOLUMN_NAME VARCHAR2(255),
			CONSTRAINT_CLAUSE VARCHAR2(255) 
		)	
	]');

	DBMS_OUTPUT.PUT_LINE('Table MD_STAGE_PG8_CONSTRAINTS created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_STAGE_PG8_CONSTRAINTS creation'); 
	
END create_MD_STAGE_PG8_CONSTRAINTS;

PROCEDURE create_MD_STAGE_PG8_INDEXES
IS
BEGIN

	EXECUTE IMMEDIATE (q'[
		CREATE TABLE MD_STAGE_PG8_INDEXES 
		(   
		    CATALOG_NAME VARCHAR2(255),
		    SCHEMA_NAME VARCHAR2(255),
		    TABLE_NAME VARCHAR2(255),
		    INDEX_NAME VARCHAR2(255),
		    COLUMN_NAME VARCHAR2(255)    
		)			
	]');

	DBMS_OUTPUT.PUT_LINE('Table MD_STAGE_PG8_INDEXES created.');

    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_STAGE_PG8_INDEXES creation'); 

END create_MD_STAGE_PG8_INDEXES;

PROCEDURE create_MD_STAGE_PG8_COMMENTS
IS
BEGIN

	EXECUTE IMMEDIATE (q'[
		CREATE TABLE MD_STAGE_PG8_COMMENTS 
		(   
		    CATALOG_NAME VARCHAR2(255), 
		    SCHEMA_NAME VARCHAR2(255), 
		    OBJECT_NAME VARCHAR2(255), 
		    PARENT_NAME VARCHAR2(255), 
		    OBJECT_TYPE VARCHAR2(255), 
		    COMMENTS_TEXT VARCHAR2(4000)
		)		
	]');

	DBMS_OUTPUT.PUT_LINE('Table MD_STAGE_PG8_COMMENTS created.');
	
    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_STAGE_PG8_COMMENTS creation'); 

END create_MD_STAGE_PG8_COMMENTS;

PROCEDURE create_MD_STAGE_CONSTRAINTS
IS
BEGIN
    -- Creating a Stage table without constraints 
	EXECUTE IMMEDIATE (q'[
		CREATE TABLE MD_STAGE_CONSTRAINTS 
		    (	
		    PROJECT_ID NUMBER, 
			CONNECTION_ID NUMBER, 
			CATALOG_NAME VARCHAR2(256 BYTE), 
			TYPE VARCHAR2(32 BYTE),
			CAPT_VERSION NUMBER ,
		    CONV_VERSION NUMBER,
			SCHEMA_NAME VARCHAR2(256 BYTE), 
			TABLE_NAME VARCHAR2(256 BYTE), 
			CONSTRAINT_NAME VARCHAR2(256 BYTE), 
			COLUMN_NAME VARCHAR2(256 BYTE), 
			ORACLE_NAME VARCHAR2(256 BYTE), 
			DELETE_CLAUSE VARCHAR2(4000 BYTE), 
			CONSTRAINT_TYPE VARCHAR2(256 BYTE), 
			DETAIL_ORDER NUMBER, 
			REFCATALOG_NAME VARCHAR2(256 BYTE), 
			REFSCHEMA_NAME VARCHAR2(256 BYTE), 
			REFTABLE_NAME VARCHAR2(256 BYTE), 
			REFCOLUMN_NAME VARCHAR2(256 BYTE), 
			CONSTRAINT_TEXT CLOB, 
			ISRULE CHAR(1 BYTE) DEFAULT 'N', 
			ISCLUSTEREDINDEX CHAR(1 BYTE) DEFAULT 'N', 
			LANGUAGE VARCHAR2(40 BYTE), 
			COMMENTS VARCHAR2(4000 BYTE), 
			CONSTRAINT_NAMESPACE VARCHAR2(256 BYTE), 
			SECURITY_GROUP_ID NUMBER DEFAULT 0, 
			CREATED_ON DATE DEFAULT sysdate, 
			CREATED_BY VARCHAR2(255 BYTE), 
			LAST_UPDATED_ON DATE, 
			LAST_UPDATED_BY VARCHAR2(255 BYTE)
		    )
	 ]');

	DBMS_OUTPUT.PUT_LINE('Table MD_STAGE_CONSTRAINTS created.');
	
    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_STAGE_CONSTRAINTS creation'); 

END create_MD_STAGE_CONSTRAINTS;

PROCEDURE create_MD_STAGE_INDEXES
IS
BEGIN
	 EXECUTE IMMEDIATE (q'[
		CREATE TABLE MD_STAGE_INDEXES 
		    (	
		    PROJECT_ID NUMBER, 
			CONNECTION_ID NUMBER, 
			CATALOG_NAME VARCHAR2(256 BYTE), 
			TYPE VARCHAR2(32 BYTE),
			CAPT_VERSION NUMBER ,
		    CONV_VERSION NUMBER,
			SCHEMA_NAME VARCHAR2(256 BYTE), 
			TABLE_NAME VARCHAR2(256 BYTE), 
			COLUMN_NAME VARCHAR2(256 BYTE), 
			INDEX_NAME VARCHAR2(256 BYTE), 
			ORACLE_NAME VARCHAR2(256 BYTE), 
			INDEX_TYPE VARCHAR2(256 BYTE), 
			INDEX_PORTION NUMBER, 
			IS_DESCENDING CHAR(1 BYTE) DEFAULT 'N', 
			DETAIL_ORDER NUMBER, 
			NATIVE_SQL CLOB, 
			NATIVE_KEY VARCHAR2(4000 BYTE), 
			ISCLUSTEREDINDEX CHAR(1 BYTE) DEFAULT 'N', 
			COMMENTS VARCHAR2(4000 BYTE), 
			INDEX_NAMESPACE VARCHAR2(256 BYTE), 
			SECURITY_GROUP_ID NUMBER DEFAULT 0, 
			CREATED_ON DATE DEFAULT sysdate, 
			CREATED_BY VARCHAR2(255 BYTE), 
			LAST_UPDATED_ON DATE, 
			LAST_UPDATED_BY VARCHAR2(4000 BYTE)
		)
	 ]');

	DBMS_OUTPUT.PUT_LINE('Table MD_STAGE_INDEXES created.');
	
    EXCEPTION
            WHEN OTHERS THEN
            DBMS_OUTPUT.PUT_LINE('ERROR : Table MD_STAGE_INDEXES creation'); 

END create_MD_STAGE_INDEXES;

PROCEDURE createCaptureStageTables
IS
BEGIN

	create_MD_STAGE_PG8_DATABASES;
	create_MD_STAGE_PG8_TABLES;
	create_MD_STAGE_PG8_COLUMNS;
	create_MD_STAGE_PG8_CONSTRAINTS;
	create_MD_STAGE_PG8_INDEXES;
	create_MD_STAGE_PG8_COMMENTS;

END createCaptureStageTables;

PROCEDURE dropCaptureStageTables
IS
	CURSOR drop_table IS SELECT 'DROP TABLE '||table_name||' CASCADE CONSTRAINTS' query_item FROM USER_TABLES 
	WHERE table_name IN  
	(
	'MD_STAGE_PG8_COMMENTS',
	'MD_STAGE_PG8_INDEXES',
	'MD_STAGE_PG8_CONSTRAINTS',
	'MD_STAGE_PG8_COLUMNS',
	'MD_STAGE_PG8_TABLES',
	'MD_STAGE_PG8_DATABASES'
	);

BEGIN
FOR item IN drop_table
  LOOP
    EXECUTE IMMEDIATE item.query_item;
    DBMS_OUTPUT.PUT_LINE('Table ' || REGEXP_REPLACE(item.query_item, '^DROP\s+TABLE\s+((\w|\W)+)\s+CASCADE\s+CONSTRAINTS$', '\1') ||' dropped.');
  END LOOP;
  
END dropCaptureStageTables;

PROCEDURE truncateCaptureStageTables
IS
	-- CURSORS --
	CURSOR truncate_table IS SELECT 'TRUNCATE TABLE '||table_name query_item FROM USER_TABLES 
	WHERE table_name IN  
	(
	'MD_STAGE_PG8_COMMENTS',
	'MD_STAGE_PG8_INDEXES',
	'MD_STAGE_PG8_CONSTRAINTS',
	'MD_STAGE_PG8_COLUMNS',
	'MD_STAGE_PG8_TABLES',
	'MD_STAGE_PG8_DATABASES'
	);

BEGIN
	FOR item IN truncate_table
	  LOOP
	    EXECUTE IMMEDIATE item.query_item;
	    DBMS_OUTPUT.PUT_LINE('TRUNCATE TABLE ' || REGEXP_REPLACE(item.query_item, '^TRUNCATE\s+TABLE\s+((\w|\W)+)', '\1'));
	  END LOOP;
	  
	COMMIT;

END truncateCaptureStageTables;

PROCEDURE createConvertStageTables
IS
BEGIN

	create_MD_STAGE_CONSTRAINTS;
	create_MD_STAGE_INDEXES;

END createConvertStageTables;

PROCEDURE dropConvertStageTables
IS
	CURSOR drop_table IS SELECT 'DROP TABLE '||table_name||' CASCADE CONSTRAINTS' query_item FROM USER_TABLES 
	WHERE table_name IN  
	(
	'MD_STAGE_CONSTRAINTS',
	'MD_STAGE_INDEXES'
	);
BEGIN
FOR item IN drop_table
  LOOP
    EXECUTE IMMEDIATE item.query_item;
    DBMS_OUTPUT.PUT_LINE('Table ' || REGEXP_REPLACE(item.query_item, '^DROP\s+TABLE\s+((\w|\W)+)\s+CASCADE\s+CONSTRAINTS$', '\1') ||' dropped.');
  END LOOP;
  
END dropConvertStageTables;

PROCEDURE truncateConvertStageTables
IS
	-- CURSORS --
	CURSOR truncate_table IS SELECT 'TRUNCATE TABLE '||table_name query_item FROM USER_TABLES 
	WHERE table_name IN  
	(
	'MD_STAGE_CONSTRAINTS',
	'MD_STAGE_INDEXES'
	);

BEGIN
	FOR item IN truncate_table
	  LOOP
	    EXECUTE IMMEDIATE item.query_item;
	    DBMS_OUTPUT.PUT_LINE('TRUNCATE TABLE ' || REGEXP_REPLACE(item.query_item, '^TRUNCATE\s+TABLE\s+((\w|\W)+)', '\1'));
	  END LOOP;
	  
	COMMIT;

END truncateConvertStageTables;

PROCEDURE createAllRepositoryTables
IS
BEGIN
  	createRepositoryTables;
	createCaptureStageTables;
	createConvertStageTables;

END createAllRepositoryTables;

PROCEDURE dropAllRepositoryTables
IS
BEGIN
   	dropRepositoryTables;
	dropCaptureStageTables;
	dropConvertStageTables;
    dropMigrationPackages;

END dropAllRepositoryTables;

PROCEDURE truncateAllRepositoryTables
IS
BEGIN
 	truncateRepositoryTables;
	truncateCaptureStageTables;
	truncateConvertStageTables;

END truncateAllRepositoryTables;

/*
Renamed object names 

'MIGR_DATATYPE_TRANSFORM_MAP'  -> 'MD_DATATYPE_TRANSFORM_MAP',
'MIGR_DATATYPE_TRANSFORM_RULE' -> 'MD_DATATYPE_TRANSFORM_RULE'
'MIGR_GENERATION_ORDER'        -> 'MD_GENERATION_ORDER'
'MIGRATION_RESERVED_WORDS'     -> 'MD_RESERVED_WORDS'
'MIGRLOG'                      -> 'MD_MIGRLOG'

MIGR_MIGRLOG_FK  ->  MD_MIGRLOG_FK

PACKAGE SESSIONSTATE   -> PACKAGE MIGRATION_SESSIONSTATE

*/

END;
/
