/* iod-telemetry ActiveSessionHistory.sql version 2022.01.12 stkolev */
/* Notes:
    1. There are some sql_ids in ASH that don't exist in sqlstat. This causes ActiveSessionHistory.SqlId.Sessions1mAvg to return about 0.5 - 1 less AAS than the rest.
       Performing an outer join to sqlstats has a huge performance penatly and returns more AAS than the rest
    2. Some applications use literals in their sql's and as a result a lot of sql statements are sent to t2. This causes severe performance issues for Grafana
       because it has to load a lot of sql_id's and sql_text's into the browser memory.
       To address this:
            - ActiveSessionHistory.SqlId.Sessions1mAvg groups AAS <= x/60 into an "other" sql_id, sql_type, and sql_text
            - ActiveSessionHistory.Sqlstats.* returns statistics for sqls that have AAS >= x/60
       where x is the number of times the sql shows up in the 60 second ASH sample. This also addresses infrequently executed sql's
    3. Some sql_id's are null. To address this
        Return module=<module> for sql_text if module is not null
        Return program=<program> for sql_text if module is null
*/
WITH
FUNCTION application_category (
  p_sql_text     IN VARCHAR2, 
  p_command_name IN VARCHAR2 DEFAULT NULL
)
RETURN VARCHAR2 DETERMINISTIC
IS
  k_appl_handle_prefix CONSTANT VARCHAR2(30) := '/*'||CHR(37);
  k_appl_handle_suffix CONSTANT VARCHAR2(30) := CHR(37)||'*/'||CHR(37);
BEGIN
  IF    p_sql_text LIKE k_appl_handle_prefix||'Transaction Processing'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'addTransactionRow'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'checkEndRowValid'||k_appl_handle_suffix
    OR  p_sql_text LIKE k_appl_handle_prefix||'checkStartRowValid'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'deleteValue'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'exists'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch commit by idempotency token'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch latest transactions for cache'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Find lower commit id for transaction cache warm up'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'findMatchingRow'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getMaxTransactionCommitID'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getNewTransactionID'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getTransactionProgress'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'lockForCommit'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'lockKievTransactor'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'putBucket'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'readTransactionsSince'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'recordTransactionState'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'setValue'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'SPM:CP'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateIdentityValue'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateNextKievTransID'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateTransactorState'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'upsert_transactor_state'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'writeTransactionKeys'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'QueryTransactorHosts'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'WriteBucketValues'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'batch commit'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'batch mutation log'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'fetchAllIdentities'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'fetch_epoch'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'readFromTxorStateBeginTxn'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'readOnlyBeginTxn'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'validateTransactorState'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getDataStoreMaxTransaction'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'legacyGetDataStoreMaxTransaction'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'isPartitionDropDisabled'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getWithVersionOffsetSql'||k_appl_handle_suffix 
    OR  LOWER(p_sql_text) LIKE CHR(37)||'lock table kievtransactions'||CHR(37) 
    --
    OR  p_sql_text LIKE k_appl_handle_prefix||'Set ddl lock timeout for session'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.leases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.step_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.historical_assignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.workflow_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.step_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'delete.leases_types'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getForUpdate.dataplane_alias'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.leases_types'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.leases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.step_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.historical_assignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.workflow_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert.step_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.dataplane_alias'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.leases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.step_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.historical_assignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update.workflow_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Drop partition'||k_appl_handle_suffix 
  THEN RETURN 'TP'; /* Transaction Processing */
  --
  ELSIF p_sql_text LIKE k_appl_handle_prefix||'Read Only'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'bucketIndexSelect'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'bucketKeySelect'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'bucketValueSelect'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countTransactions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch snapshots'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Get system time'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getAutoSequences'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getNextIdentityValue'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getValues'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Lock row Bucket_Snapshot'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'longFromDual'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'performContinuedScanValues'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'performFirstRowsScanQuery'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'performScanQuery'||k_appl_handle_suffix
    OR  p_sql_text LIKE k_appl_handle_prefix||'performSnapshotScanQuery'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'performStartScanValues'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'selectBuckets'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch latest revisions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch max sequence for'||k_appl_handle_suffix -- streaming
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch partition interval for'||k_appl_handle_suffix -- streaming
    OR  p_sql_text LIKE k_appl_handle_prefix||'Find High value for'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Find partitions for'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Init lock name for snapshot'||k_appl_handle_suffix -- snapshot
    OR  p_sql_text LIKE k_appl_handle_prefix||'List snapshot tables.'||k_appl_handle_suffix -- snapshot
    OR  p_sql_text LIKE k_appl_handle_prefix||'Tail read bucket'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'performSegmentedScanQuery'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'listArchiveStatusByIndexName'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'listAssignmentsByIndexName'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'listHosts'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateHost'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateArchiveStatus'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateOperationLock'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get KIEVWORKFLOWS table indexes'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'GetOldestLsn'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'GetStreamRecords'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Check if another workflow is running'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Delete old workflows from'||k_appl_handle_suffix 
    --
    OR  p_sql_text LIKE k_appl_handle_prefix||'getFutureWorkflowDefinition'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getPriorWorkflowDefinition'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'enumerateLeases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'enumerateLeaseTypes'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getHistoricalAssignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getAllWorkflowDefinitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getVersionHistory'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getInstances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Find interval partitions for schema'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getStepInstances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getOldestGcWorkflowInstance'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getNextRecordID'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.dataplane_alias'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getLeaseNonce'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.leases_types'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.leases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getByKey.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getRecordId.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getLast.historical_assignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.historical_assignments'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'isPartitionDropDisabled'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Check if there are active rows for partition'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getRunningInstancesCount'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.workflow_definitions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getLatestWorkflowDefinition'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getLast.step_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.workflow_instances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get.step_instances'||k_appl_handle_suffix 
  THEN RETURN 'RO'; /* Read Only */
  --
  ELSIF p_sql_text LIKE k_appl_handle_prefix||'Background'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Bootstrap snapshot table Kiev_S'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'bucketIdentitySelect'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'checkMissingTables'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countAllBuckets'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countAllRows'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countKievTransactionRows'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countKtkRows'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Delete garbage'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Delete rows from'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'deleteBucketGarbage'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'enumerateSequences'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch config'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'fetch_leader_heartbeat'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'gcEventMaxId'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'gcEventTryInsert'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Get txn at time'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get_leader'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getCurEndTime'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getDBSchemaVersion'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getEndTimeOlderThan'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getGCLogEntries'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getMaxTransactionOlderThan'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getSchemaMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getSupportedLibVersions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'hashBucket'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'hashSnapshot'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Populate workspace'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'populateBucketGCWorkspace'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'primeTxCache'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'readOnlyRoleExists'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Row count between transactions'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'secondsSinceLastGcEvent'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'sync_leadership'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Test if table Kiev_S'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Update snapshot metadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'update_heartbeat'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'validateIfWorkspaceEmpty'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'verify_is_leader'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Checking existence of Mutation Log Table'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Checks if KIEVTRANSACTIONKEYS table is empty'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Checks if KIEVTRANSACTIONS table is empty'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch partition interval for KT'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Fetch partition interval for KTK'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'Insert dynamic config'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'createProxyUser'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'createSequence'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'deregister_host'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'dropAutoSequenceMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'dropBucketFromMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'dropSequenceMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get KievTransactionKeys table indexes'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get KievTransactions table indexes'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'get session count'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'initializeMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'isKtPartitioned'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'isPartitioned'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'log'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'register_host'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'updateSchemaVersionInDB'||k_appl_handle_suffix 
    --
    OR  p_sql_text LIKE k_appl_handle_prefix||'getUnownedLeases'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getFutureWorks'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getMinorVersionsAtAndAfter'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getLeaseDecorators'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getUnownedLeasesByFiFo'||k_appl_handle_suffix 
  THEN RETURN 'BG'; /* Background */
  --
  ELSIF p_sql_text LIKE k_appl_handle_prefix||'Ignore'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'enumerateKievPdbs'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'getJDBCSuffix'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'MV_REFRESH'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'null'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'selectColumnsForTable'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'selectDatastoreMd'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'SQL Analyze('||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'validateDataStoreId'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countMetadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'countSequenceInstances'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'iod-telemetry'||k_appl_handle_suffix 
    OR  p_sql_text LIKE k_appl_handle_prefix||'insert snapshot metadata'||k_appl_handle_suffix 
    OR  p_sql_text LIKE CHR(37)||k_appl_handle_prefix||'OPT_DYN_SAMP'||k_appl_handle_suffix 
  THEN RETURN 'IG'; /* Ignore */
  --
  ELSIF p_command_name IN ('INSERT', 'UPDATE')
  THEN RETURN 'TP'; /* Transaction Processing */
  --
  ELSIF p_command_name = 'DELETE'
  THEN RETURN 'BG'; /* Background */
  --
  ELSIF p_command_name = 'SELECT'
  THEN RETURN 'RO'; /* Read Only */
  --
  ELSE RETURN 'UN'; /* Unknown */
  END IF;
END application_category;
    /****************************************************************************************/
    v$metricname as (
        /* version 3 */
        select
            ''              as project
            ,'oracle_db'    as fleet
            ,host_name      as host
            ,'"lifecycle":"'||
                case
                    when
                        (regexp_like(instance_name,'(9[0-9])') and host_name not like '%.r1') or
                        regexp_like(host_name,'(dev|sandbox)') or
                        regexp_like(lower(sys_context('USERENV','DB_DOMAIN')),'(dev|sand)')
                        then 'DEV'
                    when
                        (regexp_like(instance_name,'(9[0-9])') and host_name like '%.r1%') or
                        regexp_like(host_name,'(test|tst)') or
                        regexp_like(lower(sys_context('USERENV','DB_DOMAIN')),'(test|tst)')
                        then 'TEST'
                    else 'PROD'
                end || '"' ||
            ',"hostclass":"'|| nvl(regexp_substr(host_name,'^([^.]*)-',1,1,NULL,1),'N/A') || '"' ||
            ',"instance_name":"'     || instance_name    || '"' ||
            ',"db_name":"'           || nvl(name,'N/A')             || '"' ||
            ',"db_domain":"'         || nvl(sys_context('USERENV','DB_DOMAIN'),'N/A') || '"' ||
            ',"db_unique_name":"'    || db_unique_name   || '"' ||
            ',"database_role":"'     || database_role    || '"' ||
            ',"fsfo_role":"'||
                case
                    when database_role = 'PRIMARY' then 'PRIMARY'
                    when database_role = 'PHYSICAL STANDBY' and lower(db_unique_name)  = lower(fs_failover_current_target) then 'STANDBY'
                    when database_role = 'PHYSICAL STANDBY' and lower(db_unique_name) != lower(fs_failover_current_target) then 'BYSTANDER'
                    else 'NONE'
                end || '"' as metric_dimensions
        from v$database, v$instance
    ),
    v$metric_history as (
        select /*+ MATERIALIZE NO_MERGE */
             con.con_id
            ,con.name as con_name
            ,nvl(ash.sql_id,ash.top_level_sql_id) sql_id
            ,ash.module
            ,ash.program
            ,ash.machine
            ,ash.sql_opname
            ,case when ash.session_state = 'ON CPU' then 'CPU' else ash.wait_class end as wait_class
            ,case when ash.session_state = 'ON CPU' then 'CPU' else ash.event end as event
            ,regexp_substr(coalesce(b.machine,s.machine),'((?:[^.]*\.){0}[^.]*)') as blocking_host
        from
             v$active_session_history ash
            ,v$session s
            ,v$session b
            ,v$containers con
        where
                ash.con_id (+) = con.con_id
            and ash.blocking_session = s.sid (+)
            and ash.blocking_session_serial# = s.serial# (+)
            and s.final_blocking_session = b.sid (+)
            and b.final_blocking_session_status (+) = 'VALID'
            and ash.sample_time > systimestamp - interval '60' second
    ),
    v$metric_group as (
        select /*+ MATERIALIZE NO_MERGE */
             ash.con_name
            ,ash.con_id
            ,ash.sql_id
            ,ash.wait_class
            ,ash.event
            ,ash.sql_opname
            ,sql.ExecsPerMinute
            ,sql.ElapsedTimeMsPerExec
            ,sql.CpuTimeMsPerExec
            ,sql.IoWaitMsPerExec
            ,sql.ClWaitMsPerExec
            ,sql.ApWaitMsPerExec
            ,sql.CcWaitMsPerExec
            ,sql.RowsProcessedPerExec
            ,sql.BufferGetsPerExec
            ,sql.DiskReadsPerExec
            ,sql.DirectWritesPerExec
            ,sql.FetchCountPerExec
            ,sql.WaitsOverheadPerc
            ,sql.DiskIoOverheadPerc
            --,sql.sql_type
            ,application_category(sql.sql_text, ash.sql_opname) as sql_type
            ,sql.sql_text
        from
             v$metric_history ash
            ,(
                SELECT
                    s.con_id
                    ,s.sql_id
                    ,ROUND(s.delta_execution_count / t.minutes,3)                                     as ExecsPerMinute
                    ,ROUND(s.delta_elapsed_time/GREATEST(s.delta_execution_count,1)/1e3,3)            as ElapsedTimeMsPerExec
                    ,ROUND(s.delta_cpu_time/GREATEST(s.delta_execution_count,1)/1e3,3)                as CpuTimeMsPerExec
                    ,ROUND(s.delta_user_io_wait_time/GREATEST(s.delta_execution_count,1)/1e3,3)       as IoWaitMsPerExec
                    ,ROUND(s.delta_cluster_wait_time/GREATEST(s.delta_execution_count,1)/1e3,3)       as ClWaitMsPerExec
                    ,ROUND(s.delta_application_wait_time/GREATEST(s.delta_execution_count,1)/1e3,3)   as ApWaitMsPerExec
                    ,ROUND(s.delta_concurrency_time/GREATEST(s.delta_execution_count,1)/1e3,3)        as CcWaitMsPerExec
                    ,ROUND(s.delta_rows_processed/GREATEST(s.delta_execution_count,1),3)              as RowsProcessedPerExec
                    ,ROUND(s.delta_buffer_gets/GREATEST(s.delta_execution_count,1),3)                 as BufferGetsPerExec
                    ,ROUND(s.delta_disk_reads/GREATEST(s.delta_execution_count,1),3)                  as DiskReadsPerExec
                    ,ROUND(s.delta_direct_writes/GREATEST(s.delta_execution_count,1),3)               as DirectWritesPerExec
                    ,ROUND(s.delta_fetch_count/GREATEST(s.delta_execution_count,1),3)                 as FetchCountPerExec
                    ,ROUND(100*(1-(s.delta_cpu_time/GREATEST(s.delta_elapsed_time,1))),3)             as WaitsOverheadPerc
                    ,ROUND(100*s.delta_disk_reads/GREATEST(s.delta_buffer_gets,1),3)                  as DiskIoOverheadPerc
                    --,application_category(s.sql_text)                                                 as sql_type
                    ,regexp_replace(replace(replace(replace(substr(s.sql_text,1,500),chr(39)),'\','\\'),'"','\"'),'[^[:print:]]','') as sql_text
                  FROM v$sqlstats s,
                      (  SELECT GREATEST((SYSDATE - CAST(MAX(end_interval_time) AS DATE)) * 24 * 60, (1/60)) minutes
                           FROM dba_hist_snapshot
                      ) t
              ) sql
          where
              ash.con_id = sql.con_id and
              ash.sql_id = sql.sql_id
    ),
    v$metric as (
        select
            'ActiveSessionHistory.ConName.Sessions1mAvg' as metric_name
            ,',"con_name":"'    || nvl(to_char(con_name     ),'null') || '"' as metric_dimensions
             ,round(count(*)/60,2) as metric_value
         from
            v$metric_history
        group by
            con_name
        union
        select
            'ActiveSessionHistory.WaitClass.Sessions1mAvg' as metric_name
            ,',"con_name":"'    || nvl(to_char(con_name     ),'null') || '"' ||
             ',"wait_class":"'  || nvl(to_char(wait_class   ),'null') || '"' as metric_dimensions
             ,round(count(*)/60,2) as metric_value
         from
            v$metric_history
        group by
            con_name
            ,wait_class
        union
        select
            'ActiveSessionHistory.Event.Sessions1mAvg' as metric_name
            ,',"con_name":"'    || nvl(to_char(con_name     ),'null') || '"' ||
             ',"event":"'       || nvl(to_char(event        ),'null') || '"' ||
             ',"wait_class":"'  || nvl(to_char(wait_class   ),'null') || '"' as metric_dimensions
             ,round(count(*)/60,2) as metric_value
         from
            v$metric_history
        group by
            con_name
            ,event
            ,wait_class
        union
        select
            'ActiveSessionHistory.SqlId.Sessions1mAvg' as metric_name
            ,',"con_name":"'    || nvl(to_char(con_name     ),'null') || '"' ||
             ',"sql_id":"'      || nvl(to_char(sql_id       ),'null') || '"' ||
             ',"sql_opname":"'  || nvl(to_char(sql_opname   ),'null') || '"' ||
             ',"sql_type":"'    || nvl(to_char(sql_type     ),'null') || '"' ||
             ',"sql_text":"'    || nvl(to_char(sql_text     ),'null') || '"' as metric_dimensions
            ,round(sum(aas),2) as metric_value
        from (
            select
                 con_name
                ,case when count(*)/60 <= 2/60 then 'other' else sql_id     end  as sql_id
                ,case when count(*)/60 <= 2/60 then 'other' else sql_opname end  as sql_opname
                ,case when count(*)/60 <= 2/60 then 'other' else sql_type   end  as sql_type
                ,case when count(*)/60 <= 2/60 then 'other' else sql_text   end  as sql_text
                ,count(*)/60                                                     as aas
            from (
                select
                    con_name, sql_id, sql_opname, sql_type, sql_text
                from v$metric_group
                union all
                select
                    con_name ,null, sql_opname ,null
                    ,substr(replace(replace(nvl2(module,'module='||regexp_replace(module,'@'||machine||' ',' '),'program='||regexp_replace(program,'@'||machine||' ',' ')),'\','\\'),'"','\"'),1,60)
                from v$metric_history where sql_id is null
            )
            group by
                con_name
                ,sql_id
                ,sql_opname
                ,sql_type
                ,sql_text
        )
        group by
            con_name
            ,sql_id
            ,sql_opname
            ,sql_type
            ,sql_text
        union
        select
            'ActiveSessionHistory.BlockingHost.Sessions1mAvg' as metric_name
            ,',"con_name":"'        || nvl(to_char(con_name         ),'null') || '"' ||
             ',"blocking_host":"'   || nvl(to_char(blocking_host    ),'null') || '"' as metric_dimensions
             ,round(count(*)/60,2) as metric_value
        from
            v$metric_history
        where
            blocking_host is not null
        group by
            con_name
            ,blocking_host
        union
        select
            'ActiveSessionHistory.Sqlstats.' || metric_name as metric_name
            ,',"con_name":"'     || nvl(to_char(con_name    ),'null') || '"' ||
             ',"sql_id":"'       || nvl(to_char(sql_id      ),'null') || '"' ||
             ',"sql_opname":"'   || nvl(to_char(sql_opname  ),'null') || '"' ||
             ',"sql_type":"'     || nvl(to_char(sql_type    ),'null') || '"' as metric_dimensions
            ,round(metric_value,2) as metric_value
         from (
            select
                con_name
                ,sql_id
                ,sql_opname
                ,sql_type
                ,sql_text
                ,max(ExecsPerMinute)        as "ExecsPerMinute"
                ,max(ElapsedTimeMsPerExec)  as "ElapsedTimeMsPerExec"
                ,max(CpuTimeMsPerExec)      as "CpuTimeMsPerExec"
                ,max(IoWaitMsPerExec)       as "IoWaitMsPerExec"
                ,max(ClWaitMsPerExec)       as "ClWaitMsPerExec"
                ,max(ApWaitMsPerExec)       as "ApWaitMsPerExec"
                ,max(CcWaitMsPerExec)       as "CcWaitMsPerExec"
                ,max(RowsProcessedPerExec)  as "RowsProcessedPerExec"
                ,max(BufferGetsPerExec)     as "BufferGetsPerExec"
                ,max(DiskReadsPerExec)      as "DiskReadsPerExec"
                ,max(DirectWritesPerExec)   as "DirectWritesPerExec"
                ,max(FetchCountPerExec)     as "FetchCountPerExec"
                ,max(WaitsOverheadPerc)     as "WaitsOverheadPerc"
                ,max(DiskIoOverheadPerc)    as "DiskIoOverheadPerc"
            from
                v$metric_group
            group by
                con_name
                ,sql_id
                ,sql_opname
                ,sql_type
                ,sql_text
            having count(*)/60 >= 2/60
        ) unpivot (
            metric_value FOR metric_name IN (
                 "ExecsPerMinute"
                ,"ElapsedTimeMsPerExec"
                ,"CpuTimeMsPerExec"
                ,"IoWaitMsPerExec"
                ,"ClWaitMsPerExec"
                ,"ApWaitMsPerExec"
                ,"CcWaitMsPerExec"
                ,"RowsProcessedPerExec"
                ,"BufferGetsPerExec"
                ,"DiskReadsPerExec"
                ,"DirectWritesPerExec"
                ,"FetchCountPerExec"
                ,"WaitsOverheadPerc"
                ,"DiskIoOverheadPerc"
            )
        )
    )
select
    v$metricname.project                        as project
    ,v$metricname.fleet                         as fleet
    ,v$metricname.host                          as host
    ,v$metric.metric_name                       as metric_name
    ,v$metric.metric_value                      as metric_value
    ,'{'||v$metricname.metric_dimensions||v$metric.metric_dimensions||'}' as metric_dimensions
from v$metricname,v$metric
