package org.jdesktop.animation.timing.interpolation;

import java.lang.reflect.Method;

public class KeyFrames
{

    public enum InterpolationType
    {
        LINEAR, 
        DISCRETE, 
        NONLINEAR;
    }

    private KeyValues keyValues;
    private KeyTimes keyTimes;
    private KeySplines keySplines;
    private InterpolationType interpolationType;
    
    public KeyFrames(final KeyValues keyValues) {
        this.init(keyValues, null, null, InterpolationType.LINEAR);
    }
    
    public KeyFrames(final KeyValues keyValues, final InterpolationType interpolationType) {
        this.init(keyValues, null, null, interpolationType);
    }
    
    public KeyFrames(final KeyValues keyValues, final KeyTimes keyTimes, final InterpolationType interpolationType) {
        this.init(keyValues, null, keyTimes, interpolationType);
    }
    
    public KeyFrames(final KeyValues keyValues, final KeySplines keySplines, final KeyTimes keyTimes, final InterpolationType interpolationType) {
        this.init(keyValues, keySplines, keyTimes, interpolationType);
    }
    
    public KeyFrames(final KeyValues keyValues, final KeySplines keySplines) {
        if (keySplines != null) {
            this.init(keyValues, keySplines, null, InterpolationType.NONLINEAR);
        }
        else {
            this.init(keyValues, keySplines, null, InterpolationType.LINEAR);
        }
    }
    
    private void init(final KeyValues keyValues, final KeySplines keySplines, final KeyTimes keyTimes, final InterpolationType interpolationType) {
        if (keyTimes == null) {
            final int numKeyTimes = keyValues.getSize();
            final float[] keyTimesArray = new float[numKeyTimes];
            float timeVal = 0.0f;
            keyTimesArray[0] = timeVal;
            for (int i = 1; i < numKeyTimes - 1; ++i) {
                timeVal += 1.0f / (numKeyTimes - 1);
                keyTimesArray[i] = timeVal;
            }
            keyTimesArray[numKeyTimes - 1] = 1.0f;
            this.keyTimes = new KeyTimes(keyTimesArray);
        }
        else {
            this.keyTimes = keyTimes;
        }
        this.keyValues = keyValues;
        this.keySplines = keySplines;
        this.interpolationType = interpolationType;
        if (interpolationType == InterpolationType.NONLINEAR && keySplines == null) {
            throw new IllegalArgumentException("NONLINEAR interpolation requires KeySplines");
        }
        if (keyValues.getSize() != this.keyTimes.getSize()) {
            throw new IllegalArgumentException("keyValues and keyTimes must be of equal size");
        }
        if (keySplines != null && keySplines.getSize() != this.keyTimes.getSize() - 1) {
            throw new IllegalArgumentException("keySplines must have a size equal to the one less than the size of keyValues");
        }
    }
    
    public Class getType() {
        return this.keyValues.getType();
    }
    
    KeyValues getKeyValues() {
        return this.keyValues;
    }
    
    KeySplines getKeySplines() {
        return this.keySplines;
    }
    
    KeyTimes getKeyTimes() {
        return this.keyTimes;
    }
    
    public void setValue(final Object object, final Method method, final float fraction) {
        final int interval = this.keyTimes.getInterval(fraction);
        final float t0 = this.keyTimes.getTime(interval);
        if (this.interpolationType == InterpolationType.DISCRETE) {
            if (fraction < 1.0f) {
                this.keyValues.setValue(object, method, interval);
            }
            else {
                this.keyValues.setValue(object, method, this.keyTimes.getSize() - 1);
            }
        }
        else {
            final float t2 = this.keyTimes.getTime(interval + 1);
            float t3 = (fraction - t0) / (t2 - t0);
            if (this.interpolationType == InterpolationType.NONLINEAR) {
                t3 = this.keySplines.interpolate(interval, t3);
            }
            this.keyValues.setValue(object, method, interval, interval + 1, t3);
        }
    }
    
}
