/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.bmc.http.signing.internal;

import com.oracle.bmc.http.client.Serializer;
import com.oracle.bmc.http.client.io.DuplicatableInputStream;
import com.oracle.bmc.http.signing.RequestSigner;
import com.oracle.bmc.http.signing.RequestSignerException;
import com.oracle.bmc.http.signing.SigningStrategy;
import com.oracle.bmc.http.signing.internal.Algorithm;
import com.oracle.bmc.http.signing.internal.KeySupplier;
import com.oracle.bmc.http.signing.internal.SignatureSigner;
import com.oracle.bmc.http.signing.internal.SignedRequestException;
import com.oracle.bmc.http.signing.internal.SignedRequestVersion;
import com.oracle.bmc.http.signing.internal.Version;
import com.oracle.bmc.io.internal.KeepOpenInputStream;
import com.oracle.bmc.retrier.Retriers;
import com.oracle.bmc.util.StreamUtils;
import com.oracle.bmc.util.VisibleForTesting;
import com.oracle.bmc.util.internal.StringUtils;
import com.oracle.bmc.util.internal.Validate;
import jakarta.annotation.Nonnull;
import jakarta.annotation.Nullable;
import java.beans.ConstructorProperties;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.interfaces.RSAPrivateKey;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.TimeZone;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RequestSignerImpl
implements RequestSigner {
    private static final Logger LOG = LoggerFactory.getLogger(RequestSignerImpl.class);
    private static final SignatureSigner SIGNER = new SignatureSigner();
    private final KeySupplier<RSAPrivateKey> keySupplier;
    private final SigningConfiguration signingConfiguration;
    private final Supplier<String> keyIdSupplier;

    public RequestSignerImpl(@Nonnull KeySupplier<RSAPrivateKey> keySupplier, @Nonnull SigningStrategy signingStrategy, @Nonnull Supplier<String> keyIdSupplier) {
        this(keySupplier, RequestSignerImpl.toSigningConfiguration(signingStrategy), keyIdSupplier);
    }

    public RequestSignerImpl(@Nonnull KeySupplier<RSAPrivateKey> keySupplier, @Nonnull SigningConfiguration signingConfiguration, @Nonnull Supplier<String> keyIdSupplier) {
        this.keySupplier = Validate.notNull(keySupplier, "keySupplier must not be null", new Object[0]);
        this.signingConfiguration = Validate.notNull(signingConfiguration, "signingConfiguration must not be null", new Object[0]);
        this.keyIdSupplier = Validate.notNull(keyIdSupplier, "keyIdSupplier must not be null", new Object[0]);
    }

    private static SigningConfiguration toSigningConfiguration(SigningStrategy signingStrategy) {
        return new SigningConfiguration(signingStrategy.getHeadersToSign(), signingStrategy.getOptionalHeadersToSign(), signingStrategy.isSkipContentHeadersForStreamingPutRequests());
    }

    @Override
    public Map<String, String> signRequest(@Nonnull URI uri, @Nonnull String httpMethod, @Nonnull Map<String, List<String>> headers, @Nullable Object body) {
        return this.signRequest(Algorithm.RSAPSS256, uri, httpMethod, headers, body, SignedRequestVersion.getLatestVersion().getVersionName());
    }

    private Map<String, String> signRequest(Algorithm algorithm, URI uri, String httpMethod, Map<String, List<String>> headers, Object body, String versionName) {
        return RequestSignerImpl.signRequest(algorithm, uri, httpMethod, headers, body, versionName, this.keyIdSupplier.get(), this.keySupplier, this.signingConfiguration);
    }

    public static Map<String, String> signRequest(Algorithm algorithm, URI uri, String httpMethod, Map<String, List<String>> headers, Object body, String versionName, String keyId, KeySupplier<RSAPrivateKey> keySupplier, SigningConfiguration signingConfiguration) {
        Validate.notNull(algorithm, "algorithm must not be null", new Object[0]);
        Validate.notNull(uri, "uri must not be null", new Object[0]);
        Validate.notBlank(httpMethod, "httpMethod must not be null or empty", new Object[0]);
        Validate.notNull(headers, "headers must not be null", new Object[0]);
        Validate.notBlank(versionName, "versionName must not be null or empty", new Object[0]);
        try {
            Version version = RequestSignerImpl.validateVersion(versionName, algorithm);
            RSAPrivateKey key = RequestSignerImpl.getPrivateKey(keyId, keySupplier);
            String lowerHttpMethod = httpMethod.toLowerCase();
            String path = RequestSignerImpl.extractPath(uri);
            List<String> requiredHeaders = RequestSignerImpl.getRequiredSigningHeaders(lowerHttpMethod, signingConfiguration);
            List<String> optionalHeaders = RequestSignerImpl.getOptionalSigningHeaders(lowerHttpMethod, signingConfiguration);
            for (String optionalHeaderName : optionalHeaders) {
                if (headers.get(optionalHeaderName) == null) continue;
                requiredHeaders.add(optionalHeaderName);
            }
            Map<String, List<String>> existingHeaders = RequestSignerImpl.ignoreCaseHeaders(headers);
            Map<String, String> missingHeaders = RequestSignerImpl.calculateMissingHeaders(lowerHttpMethod, uri, existingHeaders, body, requiredHeaders, signingConfiguration);
            HashMap<String, List<String>> allHeaders = new HashMap<String, List<String>>();
            allHeaders.putAll(existingHeaders);
            for (Map.Entry<String, String> e : missingHeaders.entrySet()) {
                if (e.getValue() == null) {
                    if (e.getKey().equals("host") && !uri.toString().toLowerCase().startsWith("http://") && !uri.toString().toLowerCase().startsWith("https://")) {
                        throw new NullPointerException("Exception while communicating to " + uri + ". Make sure the endpoint starts with the scheme (http or https)");
                    }
                    throw new NullPointerException("Exception while communicating to " + uri + ". Expecting exactly one value for header " + e.getKey());
                }
                LOG.trace("Adding missing header '{}' = '{}'", (Object)e.getKey(), (Object)e.getValue());
                allHeaders.put(e.getKey(), Collections.unmodifiableList(Arrays.asList(e.getValue())));
            }
            String stringToSign = RequestSignerImpl.calculateStringToSign(lowerHttpMethod, path, allHeaders, requiredHeaders, headers);
            String signature = RequestSignerImpl.sign(key, algorithm, stringToSign);
            String authorizationHeader = RequestSignerImpl.calculateAuthorizationHeader(keyId, lowerHttpMethod, signature, algorithm, version.getVersionName(), requiredHeaders, optionalHeaders);
            missingHeaders.put("authorization", authorizationHeader);
            for (String headerName : requiredHeaders) {
                if (missingHeaders.containsKey(headerName) || !existingHeaders.containsKey(headerName) || existingHeaders.get(headerName).isEmpty()) continue;
                missingHeaders.put(headerName, existingHeaders.get(headerName).get(0));
            }
            return missingHeaders;
        }
        catch (Exception ex) {
            LOG.debug("Could not sign request", (Throwable)ex);
            throw new SignedRequestException(ex);
        }
    }

    private static Version validateVersion(String version, Algorithm algorithm) {
        Optional<SignedRequestVersion> oVersion = SignedRequestVersion.getVersion(version);
        if (!oVersion.isPresent()) {
            LOG.debug("Invalid version number '{}'", (Object)version);
            throw new RequestSignerException("Invalid version number");
        }
        Version srVersion = oVersion.get();
        Optional<Version.Error> errorOpt = srVersion.validateAlgorithm(algorithm);
        if (errorOpt.isPresent()) {
            LOG.debug("Signature version rule validation failed '{}'", (Object)errorOpt.get());
            throw new RequestSignerException("Version validation fails " + (Object)((Object)errorOpt.get()));
        }
        return srVersion;
    }

    private static RSAPrivateKey getPrivateKey(String keyId, KeySupplier<RSAPrivateKey> keySupplier) {
        Optional<RSAPrivateKey> keyOptional = keySupplier.supplyKey(keyId);
        if (!keyOptional.isPresent()) {
            LOG.debug("Could not find private key associated with keyId '{}'", (Object)keyId);
            throw new RequestSignerException("Could not find private key");
        }
        return keyOptional.get();
    }

    @VisibleForTesting
    static Map<String, List<String>> ignoreCaseHeaders(Map<String, List<String>> originalHeaders) {
        HashMap<String, List<String>> transformedMap = new HashMap<String, List<String>>();
        for (Map.Entry<String, List<String>> entry : originalHeaders.entrySet()) {
            transformedMap.put(entry.getKey().toLowerCase(), entry.getValue());
        }
        return transformedMap;
    }

    private static String transformHeadersToJsonString(Map<String, List<String>> headers) {
        try {
            return Serializer.getDefault().writeValueAsString(headers);
        }
        catch (IOException ex) {
            LOG.debug("Unable to serialize headers to JSON string", (Throwable)ex);
            return "UNABLE TO SERIALIZE";
        }
    }

    private static String extractPath(URI uri) {
        String path = uri.getRawPath();
        String query = uri.getRawQuery();
        if (query != null && !query.trim().isEmpty()) {
            path = path + "?" + query;
        }
        return path;
    }

    static Map<String, String> calculateMissingHeaders(String httpMethod, URI uri, Map<String, List<String>> existingHeaders, Object body, List<String> requiredHeaders, SigningConfiguration signingConfiguration) throws IOException {
        HashMap<String, String> missingHeaders = new HashMap<String, String>();
        if (RequestSignerImpl.isRequiredHeaderMissing("date", requiredHeaders, existingHeaders)) {
            missingHeaders.put("date", RequestSignerImpl.createFormatter().format(new Date()));
        }
        if (RequestSignerImpl.isRequiredHeaderMissing("host", requiredHeaders, existingHeaders)) {
            String host = uri.getHost();
            int port = uri.getPort();
            if (port != -1) {
                host = host + ":" + port;
            }
            missingHeaders.put("host", host);
        }
        boolean isPost = httpMethod.equals("post");
        boolean isPut = httpMethod.equals("put");
        boolean isPatch = httpMethod.equals("patch");
        if (!(isPut || isPost || isPatch)) {
            if (body != null) {
                throw new RequestSignerException("MUST NOT send body on non-POST/PUT/PATCH request");
            }
            return missingHeaders;
        }
        if (body instanceof InputStream && signingConfiguration.skipContentHeadersForStreamingPutRequests) {
            return missingHeaders;
        }
        byte[] bodyBytes = RequestSignerImpl.readBodyBytes(body);
        if (requiredHeaders.contains("content-type")) {
            if (!existingHeaders.containsKey("content-type")) {
                if (bodyBytes.length > 0) {
                    LOG.warn("Missing 'content-type' header, defaulting to 'application/json'");
                }
                missingHeaders.put("content-type", "application/json");
            } else {
                List<String> contentTypes = existingHeaders.get("content-type");
                if (contentTypes.size() != 1) {
                    throw new IllegalArgumentException("Expected exactly one 'content-type header (received " + contentTypes.size() + ")");
                }
            }
        }
        if (RequestSignerImpl.isRequiredHeaderMissing("content-length", requiredHeaders, existingHeaders)) {
            missingHeaders.put("content-length", Integer.toString(bodyBytes.length));
        }
        if (RequestSignerImpl.isRequiredHeaderMissing("x-content-sha256", requiredHeaders, existingHeaders)) {
            missingHeaders.put("x-content-sha256", RequestSignerImpl.calculateBodySHA256(bodyBytes));
        }
        return missingHeaders;
    }

    private static boolean isRequiredHeaderMissing(String headerName, List<String> requiredHeaders, Map<String, ?> existingHeaders) {
        return requiredHeaders.contains(headerName) && !existingHeaders.containsKey(headerName);
    }

    private static String calculateBodySHA256(byte[] body) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] hash = digest.digest(body);
            return RequestSignerImpl.base64Encode(hash);
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    @VisibleForTesting
    static String calculateStringToSign(String httpMethod, String path, Map<String, List<String>> allHeaders, List<String> requiredHeaders, Map<String, List<String>> originalHeaders) {
        ArrayList<String> signatureParts = new ArrayList<String>();
        for (String headerName : requiredHeaders) {
            String headerValue;
            List<String> headerValues = allHeaders.get(headerName);
            if (headerValues != null && headerValues.size() != 1) {
                RequestSignerException exception = new RequestSignerException("Expecting exactly one value for header " + headerName);
                LOG.error("More than one value for header [{}] to be signed found.  All headers: {}", new Object[]{headerName, RequestSignerImpl.transformHeadersToJsonString(originalHeaders), exception});
                throw exception;
            }
            String string = headerValue = headerValues != null ? headerValues.get(0) : null;
            if (headerName.equals("(request-target)")) {
                headerValue = httpMethod + " " + path;
            }
            if (headerValue == null) {
                RequestSignerException exception = new RequestSignerException("Expecting exactly one value for header " + headerName);
                LOG.error("No header value for header [{}] to be signed found.  All headers: {}", new Object[]{headerName, RequestSignerImpl.transformHeadersToJsonString(originalHeaders), exception});
                throw exception;
            }
            signatureParts.add(String.format("%s: %s", headerName, headerValue));
        }
        return StringUtils.join(signatureParts, "\n");
    }

    private static String sign(RSAPrivateKey key, Algorithm algorithm, String stringToSign) {
        byte[] signature = SIGNER.sign(key, stringToSign.getBytes(StandardCharsets.UTF_8), algorithm.getJvmName());
        return RequestSignerImpl.base64Encode(signature);
    }

    private static String calculateAuthorizationHeader(String keyId, String httpMethod, String signature, Algorithm algorithm, String version, List<String> requiredHeaders, List<String> optionalHeaders) {
        String authorizationHeader = "Signature headers=\"%s\",keyId=\"%s\",algorithm=\"%s\",signature=\"%s\",version=\"%s\"";
        String headers = StringUtils.join(requiredHeaders, " ");
        String algorithmName = algorithm.getSpecName();
        return String.format("Signature headers=\"%s\",keyId=\"%s\",algorithm=\"%s\",signature=\"%s\",version=\"%s\"", headers, keyId, algorithmName, signature, version);
    }

    private static List<String> getRequiredSigningHeaders(String httpMethod, SigningConfiguration signingConfiguration) {
        List headerNames = (List)signingConfiguration.headersToSign.get(httpMethod);
        return RequestSignerImpl.getIgnoreCaseHeaders(headerNames);
    }

    private static List<String> getIgnoreCaseHeaders(List<String> headerNames) {
        if (headerNames == null) {
            return new ArrayList<String>(0);
        }
        ArrayList<String> result = new ArrayList<String>(headerNames.size());
        for (String headerName : headerNames) {
            result.add(headerName.toLowerCase());
        }
        return result;
    }

    private static List<String> getOptionalSigningHeaders(String httpMethod, SigningConfiguration signingConfiguration) {
        List headerNames = (List)signingConfiguration.optionalHeadersToSign.get(httpMethod);
        return RequestSignerImpl.getIgnoreCaseHeaders(headerNames);
    }

    private static byte[] readBodyBytes(Object body) throws IOException {
        if (body == null) {
            return "".getBytes(StandardCharsets.UTF_8);
        }
        if (body instanceof String) {
            return ((String)body).getBytes(StandardCharsets.UTF_8);
        }
        if (body instanceof DuplicatableInputStream) {
            InputStream duplicatedBody = ((DuplicatableInputStream)body).duplicate();
            return StreamUtils.toByteArray(duplicatedBody);
        }
        if (body instanceof KeepOpenInputStream) {
            byte[] byteArr = StreamUtils.toByteArray((KeepOpenInputStream)body);
            Retriers.tryResetStreamForRetry((InputStream)body, true);
            return byteArr;
        }
        if (body instanceof InputStream) {
            throw new IllegalArgumentException("Only DuplicatableInputStream supported for body that needs signing.");
        }
        String bodyAsString = Serializer.getDefault().writeValueAsString(body);
        if (bodyAsString.equals("\"{}\"")) {
            return "{}".getBytes();
        }
        throw new IllegalArgumentException("Unexpected body type: " + body.getClass().getName());
    }

    static String base64Encode(byte[] bytes) {
        return new String(Base64.getEncoder().encode(bytes), StandardCharsets.UTF_8);
    }

    private static SimpleDateFormat createFormatter() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z", Locale.US);
        dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
        return dateFormat;
    }

    public static class SigningConfiguration {
        private final Map<String, List<String>> headersToSign;
        private final Map<String, List<String>> optionalHeadersToSign;
        private final boolean skipContentHeadersForStreamingPutRequests;

        @ConstructorProperties(value={"headersToSign", "optionalHeadersToSign", "skipContentHeadersForStreamingPutRequests"})
        public SigningConfiguration(Map<String, List<String>> headersToSign, Map<String, List<String>> optionalHeadersToSign, boolean skipContentHeadersForStreamingPutRequests) {
            this.headersToSign = headersToSign;
            this.optionalHeadersToSign = optionalHeadersToSign;
            this.skipContentHeadersForStreamingPutRequests = skipContentHeadersForStreamingPutRequests;
        }
    }
}

